/****************************************************************************
*
*					SciTech OS Portability Manager Library
*
*  ========================================================================
*
*    The contents of this file are subject to the SciTech MGL Public
*    License Version 1.0 (the "License"); you may not use this file
*    except in compliance with the License. You may obtain a copy of
*    the License at http://www.scitechsoft.com/mgl-license.txt
*
*    Software distributed under the License is distributed on an
*    "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
*    implied. See the License for the specific language governing
*    rights and limitations under the License.
*
*    The Original Code is Copyright (C) 1991-1998 SciTech Software, Inc.
*
*    The Initial Developer of the Original Code is SciTech Software, Inc.
*    All Rights Reserved.
*
*  ========================================================================
*
* Language:		ANSI C
* Environment:	16/32 bit DOS
*
* Description:	Implementation for the OS Portability Manager Library, which
*				contains functions to implement OS specific services in a
*				generic, cross platform API. Porting the OS Portability
*				Manager library is the first step to porting any SciTech
*				products to a new platform.
*
****************************************************************************/

#include "pmapi.h"
#include "drvlib/os/os.h"
#include "ztimerc.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dos.h>
#include <conio.h>
#ifdef DJGPP
#include <unistd.h>
#include <sys/nearptr.h>
#else
#include <direct.h>
#endif
#ifdef	__BORLANDC__
#pragma	warn -par
#endif

/*--------------------------- Global variables ----------------------------*/

typedef struct {
	int		oldMode;
	int     old50Lines;
	} DOS_stateBuf;

#define	MAX_RM_BLOCKS	10

static struct {
	void	*p;
	uint	tag;
	} rmBlocks[MAX_RM_BLOCKS];

static uint	VESABuf_len = 1024;		/* Length of the VESABuf buffer		*/
static void *VESABuf_ptr = NULL;	/* Near pointer to VESABuf          */
static uint VESABuf_rseg;   		/* Real mode segment of VESABuf     */
static uint	VESABuf_roff;			/* Real mode offset of VESABuf      */
static void (PMAPIP fatalErrorCleanup)(void) = NULL;
ushort _VARAPI _PM_savedDS = 0;
#ifdef	DOS4GW
static ulong	PDB = 0,*pPDB = NULL;
#endif

/*----------------------------- Implementation ----------------------------*/

void 	MTRR_init(void);
ulong 	_ASMAPI _PM_getPDB(void);

void PMAPI PM_init(void)
{
#ifndef	REALMODE
	MTRR_init();
#endif
}

ibool PMAPI PM_haveBIOSAccess(void)
{ return true; }

long PMAPI PM_getOSType(void)
{ return _OS_DOS; }

int PMAPI PM_getModeType(void)
{
#if	defined(REALMODE)
	return PM_realMode;
#elif defined(PM286)
	return PM_286;
#elif defined(PM386)
	return PM_386;
#endif
}

void PMAPI PM_backslash(char *s)
{
	uint pos = strlen(s);
	if (s[pos-1] != '\\') {
		s[pos] = '\\';
		s[pos+1] = '\0';
		}
}

void PMAPI PM_setFatalErrorCleanup(
	void (PMAPIP cleanup)(void))
{
	fatalErrorCleanup = cleanup;
}

void PMAPI PM_fatalError(const char *msg)
{
	if (fatalErrorCleanup)
		fatalErrorCleanup();
	fprintf(stderr,"%s\n", msg);
	exit(1);
}

static void ExitVBEBuf(void)
{
	if (VESABuf_ptr)
		PM_freeRealSeg(VESABuf_ptr);
	VESABuf_ptr = 0;
}

void * PMAPI PM_getVESABuf(uint *len,uint *rseg,uint *roff)
{
	if (!VESABuf_ptr) {
		/* Allocate a global buffer for communicating with the VESA VBE */
		if ((VESABuf_ptr = PM_allocRealSeg(VESABuf_len, &VESABuf_rseg, &VESABuf_roff)) == NULL)
			return NULL;
		atexit(ExitVBEBuf);
		}
	*len = VESABuf_len;
	*rseg = VESABuf_rseg;
	*roff = VESABuf_roff;
	return VESABuf_ptr;
}

int PMAPI PM_int386(int intno, PMREGS *in, PMREGS *out)
{
	PMSREGS	sregs;
	PM_segread(&sregs);
	return PM_int386x(intno,in,out,&sregs);
}

/* Routines to set and get the real mode interrupt vectors, by making
 * direct real mode calls to DOS and bypassing the DOS extenders API.
 * This is the safest way to handle this, as some servers try to be
 * smart about changing real mode vectors.
 */

void PMAPI _PM_getRMvect(int intno, long *realisr)
{
	RMREGS	regs;
	RMSREGS	sregs;

	PM_saveDS();
	regs.h.ah = 0x35;
	regs.h.al = intno;
	PM_int86x(0x21, &regs, &regs, &sregs);
	*realisr = ((long)sregs.es << 16) | regs.x.bx;
}

void PMAPI _PM_setRMvect(int intno, long realisr)
{
	RMREGS	regs;
	RMSREGS	sregs;

	PM_saveDS();
	regs.h.ah = 0x25;
	regs.h.al = intno;
	sregs.ds = (int)(realisr >> 16);
	regs.x.dx = (int)(realisr & 0xFFFF);
	PM_int86x(0x21, &regs, &regs, &sregs);
}

void PMAPI _PM_addRealModeBlock(void *mem,uint tag)
{
	int i;

	for (i = 0; i < MAX_RM_BLOCKS; i++) {
		if (rmBlocks[i].p == NULL) {
			rmBlocks[i].p = mem;
			rmBlocks[i].tag = tag;
			return;
			}
		}
	PM_fatalError("To many real mode memory block allocations!");
}

uint PMAPI _PM_findRealModeBlock(void *mem)
{
	int i;

	for (i = 0; i < MAX_RM_BLOCKS; i++) {
		if (rmBlocks[i].p == mem)
			return rmBlocks[i].tag;
		}
	PM_fatalError("Could not find prior real mode memory block allocation!");
	return 0;
}

const char * PMAPI PM_getCurrentPath(void)
{
	static char cwd[256];
	return getcwd(cwd,sizeof(cwd));
}

char PMAPI PM_getBootDrive(void)
{ return 'C'; }

const char * PMAPI PM_getVBEAFPath(void)
{ return "c:\\"; }

const char * PMAPI PM_getNucleusPath(void)
{
	static char	path[256];
	char 		*env;

	if ((env = getenv("NUCLEUS_PATH")) != NULL)
		return env;
	if ((env = getenv("WINBOOTDIR")) != NULL) {
		strcpy(path,env);
		strcat(path,"\\system\\nucleus");
		return path;
		}
	return "c:\\nucleus";
}

const char * PMAPI PM_getNucleusConfigPath(void)
{
	static char path[256];
	strcpy(path,PM_getNucleusPath());
	PM_backslash(path);
	strcat(path,"config");
	return path;
}

const char * PMAPI PM_getUniqueID(void)
{ return "DOS"; }

const char * PMAPI PM_getMachineName(void)
{ return "DOS"; }

int PMAPI PM_kbhit(void)
{
	return kbhit();
}

int PMAPI PM_getch(void)
{
	return getch();
}

PM_HWND	PMAPI PM_openConsole(PM_HWND hwndUser,int device,int xRes,int yRes,int bpp,ibool fullScreen)
{
	/* Not used for DOS */
	(void)hwndUser;
	(void)device;
	(void)xRes;
	(void)yRes;
	(void)bpp;
	(void)fullScreen;
	return 0;
}

int PMAPI PM_getConsoleStateSize(void)
{
	return sizeof(DOS_stateBuf);
}

void PMAPI PM_saveConsoleState(void *stateBuf,PM_HWND hwndConsole)
{
	RMREGS  		regs;
	DOS_stateBuf	*sb = stateBuf;

	/* Save the old video mode state */
	regs.h.ah = 0x0F;
	PM_int86(0x10,&regs,&regs);
	sb->oldMode = regs.h.al & 0x7F;
	sb->old50Lines = false;
	if (sb->oldMode == 0x3) {
		regs.x.ax = 0x1130;
		regs.x.bx = 0;
		regs.x.dx = 0;
		PM_int86(0x10,&regs,&regs);
		sb->old50Lines = (regs.h.dl == 42 || regs.h.dl == 49);
		}
	(void)hwndConsole;
}

void PMAPI PM_changeResolution(PM_HWND hwndConsole,int xRes,int yRes,int bpp)
{
	/* Not used for DOS */
	(void)hwndConsole;
	(void)xRes;
	(void)yRes;
	(void)bpp;
}

void PMAPI PM_setSuspendAppCallback(int (_ASMAPIP saveState)(int flags))
{
	/* Not used for DOS */
	(void)saveState;
}

void PMAPI PM_restoreConsoleState(const void *stateBuf,PM_HWND hwndConsole)
{
	RMREGS  			regs;
	const DOS_stateBuf	*sb = stateBuf;

	/* Retore 50 line mode if set */
	if (sb->old50Lines) {
		regs.x.ax = 0x1112;
		regs.x.bx = 0;
		PM_int86(0x10,&regs,&regs);
		}
	(void)hwndConsole;
}

void PMAPI PM_closeConsole(PM_HWND hwndConsole)
{
	/* Not used for DOS */
	(void)hwndConsole;
}

void PMAPI PM_setOSCursorLocation(int x,int y)
{
	uchar *_biosPtr = PM_getBIOSPointer();
	PM_setByte(_biosPtr+0x50,x);
	PM_setByte(_biosPtr+0x51,y);
}

void PMAPI PM_setOSScreenWidth(int width,int height)
{
	uchar *_biosPtr = PM_getBIOSPointer();
	PM_setWord(_biosPtr+0x4A,width);
	PM_setWord(_biosPtr+0x4C,width*2);
	PM_setByte(_biosPtr+0x84,height-1);
	if (height > 25) {
		PM_setWord(_biosPtr+0x60,0x0607);
		PM_setByte(_biosPtr+0x85,0x08);
		}
	else {
		PM_setWord(_biosPtr+0x60,0x0D0E);
		PM_setByte(_biosPtr+0x85,0x016);
		}
}

void * PMAPI PM_mallocShared(long size)
{
	return malloc(size);
}

int PMAPI PM_mapShared(void *ptr)
{
	(void)ptr;
	return 0;
}

void PMAPI PM_freeShared(void *ptr)
{
	free(ptr);
}

#define	GetRMVect(intno,isr)	*(isr) = ((ulong*)rmZeroPtr)[intno]
#define	SetRMVect(intno,isr)	((ulong*)rmZeroPtr)[intno] = (isr)

ibool PMAPI PM_doBIOSPOST(
	ushort axVal,
	ulong BIOSPhysAddr,
	void *mappedBIOS,
	ulong BIOSLen)
{
	static int		firstTime = true;
	static uchar	*rmZeroPtr;
	long			Current10,Current6D,Current42;
	RMREGS			regs;
	RMSREGS			sregs;

	/* Create a zero memory mapping for us to use */
	if (firstTime) {
		rmZeroPtr = PM_mapPhysicalAddr(0,0x7FFF,true);
		firstTime = false;
		}

	/* Remap the secondary BIOS to 0xC0000 physical */
	if (BIOSPhysAddr != 0xC0000L || BIOSLen > 32768) {
		/* DOS cannot virtually remap the BIOS, so we can only work if all
		 * the secondary controllers are identical, and we then use the
		 * BIOS on the first controller for all the remaining controllers.
		 *
		 * For OS'es that do virtual memory, and remapping of 0xC0000
		 * physical (perhaps a copy on write mapping) should be all that
		 * is needed.
		 */
		return false;
		}

	/* Save current handlers of int 10h and 6Dh */
	GetRMVect(0x10,&Current10);
	GetRMVect(0x6D,&Current6D);

	/* POST the secondary BIOS */
	GetRMVect(0x42,&Current42);
	SetRMVect(0x10,Current42);	/* Restore int 10h to STD-BIOS */
	regs.x.ax = axVal;
	PM_callRealMode(0xC000,0x0003,&regs,&sregs);

	/* Restore current handlers */
	SetRMVect(0x10,Current10);
	SetRMVect(0x6D,Current6D);

	/* Second the primary BIOS mappin 1:1 for 0xC0000 physical */
	if (BIOSPhysAddr != 0xC0000L) {
		/* DOS does not support this */
		(void)mappedBIOS;
		}
	return true;
}

void PMAPI PM_sleep(ulong milliseconds)
{
	ulong			microseconds = milliseconds * 1000L;
	LZTimerObject	tm;

	LZTimerOnExt(&tm);
	while (LZTimerLapExt(&tm) < microseconds)
		;
	LZTimerOffExt(&tm);
}

int PMAPI PM_getCOMPort(int port)
{
	switch (port) {
		case 0:	return 0x3F8;
		case 1:	return 0x2F8;
		}
	return 0;
}

int PMAPI PM_getLPTPort(int port)
{
	switch (port) {
		case 0:	return 0x3BC;
		case 1: return 0x378;
		case 2:	return 0x278;
		}
	return 0;
}

PM_MODULE PMAPI PM_loadLibrary(
	const char *szDLLName)
{
	(void)szDLLName;
	return NULL;
}

void * PMAPI PM_getProcAddress(
	PM_MODULE hModule,
	const char *szProcName)
{
	(void)hModule;
	(void)szProcName;
	return NULL;
}

void PMAPI PM_freeLibrary(
	PM_MODULE hModule)
{
	(void)hModule;
}

int PMAPI PM_setIOPL(
	int level)
{
	return level;
}

/*-------------------------------------------------------------------------*/
/* Generic DPMI routines common to 16/32 bit code						   */
/*-------------------------------------------------------------------------*/

ulong PMAPI DPMI_mapPhysicalToLinear(ulong physAddr,ulong limit)
{
	PMREGS  r;
	ulong	physOfs;

	if (physAddr < 0x100000L) {
		/* We can't map memory below 1Mb, but the linear address are already
		 * mapped 1:1 for this memory anyway so we just return the base address.
		 */
		return physAddr;
		}

	/* Round the physical address to a 4Kb boundary and the limit to a
	 * 4Kb-1 boundary before passing the values to DPMI as some extenders
	 * will fail the calls unless this is the case. If we round the
	 * physical address, then we also add an extra offset into the address
	 * that we return.
	 */
	physOfs = physAddr & 4095;
	physAddr = physAddr & ~4095;
	limit = ((limit+physOfs+1+4095) & ~4095)-1;

	r.x.ax = 0x800;                 /* DPMI map physical to linear      */
	r.x.bx = physAddr >> 16;
	r.x.cx = physAddr & 0xFFFF;
	r.x.si = limit >> 16;
	r.x.di = limit & 0xFFFF;
	PM_int386(0x31, &r, &r);
	if (r.x.cflag)
		return 0xFFFFFFFFUL;
	return ((ulong)r.x.bx << 16) + r.x.cx + physOfs;
}

int PMAPI DPMI_setSelectorBase(ushort sel,ulong linAddr)
{
	PMREGS  r;

	r.x.ax = 7;                     /* DPMI set selector base address   */
	r.x.bx = sel;
	r.x.cx = linAddr >> 16;
	r.x.dx = linAddr & 0xFFFF;
	PM_int386(0x31, &r, &r);
	if (r.x.cflag)
		return 0;
	return 1;
}

ulong PMAPI DPMI_getSelectorBase(ushort sel)
{
	PMREGS  r;

	r.x.ax = 6;                     /* DPMI get selector base address   */
	r.x.bx = sel;
	PM_int386(0x31, &r, &r);
	return ((ulong)r.x.cx << 16) + r.x.dx;
}

int PMAPI DPMI_setSelectorLimit(ushort sel,ulong limit)
{
	PMREGS  r;

	r.x.ax = 8;                     /* DPMI set selector limit          */
	r.x.bx = sel;
	r.x.cx = limit >> 16;
	r.x.dx = limit & 0xFFFF;
	PM_int386(0x31, &r, &r);
	if (r.x.cflag)
		return 0;
	return 1;
}

uint PMAPI DPMI_createSelector(ulong base,ulong limit)
{
	uint	sel;
	PMREGS	r;

	/* Allocate 1 descriptor */
	r.x.ax = 0;
	r.x.cx = 1;
	PM_int386(0x31, &r, &r);
	if (r.x.cflag) return 0;
	sel = r.x.ax;

	/* Set the descriptor access rights (for a 32 bit page granular
	 * segment).
	 */
	if (limit >= 0x10000L) {
		r.x.ax = 9;
        r.x.bx = sel;
        r.x.cx = 0x40F3;
		PM_int386(0x31, &r, &r);
		}

	/* Map physical memory and create selector */
	if ((base = DPMI_mapPhysicalToLinear(base,limit)) == 0xFFFFFFFFUL)
		return 0;
	if (!DPMI_setSelectorBase(sel,base))
		return 0;
	if (!DPMI_setSelectorLimit(sel,limit))
		return 0;
	return sel;
}

void PMAPI DPMI_freeSelector(uint sel)
{
	PMREGS	r;

	r.x.ax = 1;
	r.x.bx = sel;
	PM_int386(0x31, &r, &r);
}

int PMAPI DPMI_lockLinearPages(ulong linear,ulong len)
{
	PMREGS	r;

	r.x.ax = 0x600;						/* DPMI Lock Linear Region 		*/
	r.x.bx = (linear >> 16);         	/* Linear address in BX:CX 		*/
	r.x.cx = (linear & 0xFFFF);
	r.x.si = (len >> 16);         		/* Length in SI:DI 				*/
	r.x.di = (len & 0xFFFF);
	PM_int386(0x31, &r, &r);
	return (!r.x.cflag);
}

int PMAPI DPMI_unlockLinearPages(ulong linear,ulong len)
{
	PMREGS	r;

	r.x.ax = 0x601;						/* DPMI Unlock Linear Region 	*/
	r.x.bx = (linear >> 16);         	/* Linear address in BX:CX 		*/
	r.x.cx = (linear & 0xFFFF);
	r.x.si = (len >> 16);         		/* Length in SI:DI 				*/
	r.x.di = (len & 0xFFFF);
	PM_int386(0x31, &r, &r);
	return (!r.x.cflag);
}

void * PMAPI DPMI_mapPhysicalAddr(ulong base,ulong limit,ibool isCached)
{
	PMSREGS	sregs;
	ulong	linAddr;
	ulong	DSBaseAddr;

	/* Get the base address for the default DS selector */
	PM_segread(&sregs);
	DSBaseAddr = DPMI_getSelectorBase(sregs.ds);
	if ((base < 0x100000) && (DSBaseAddr == 0)) {
		/* DS is zero based, so we can directly access the first 1Mb of
		 * system memory (like under DOS4GW).
		 */
		return (void*)base;
		}

	/* Map the memory to a linear address using DPMI function 0x800 */
	if ((linAddr = DPMI_mapPhysicalToLinear(base,limit)) == 0xFFFFFFFF) {
		if (base >= 0x100000)
			return NULL;
		/* If the linear address mapping fails but we are trying to
		 * map an area in the first 1Mb of system memory, then we must
		 * be running under a Windows or OS/2 DOS box. Under these
		 * environments we can use the segment wrap around as a fallback
		 * measure, as this does work properly.
		 */
		linAddr = base;
		}

	/* Now expand the default DS selector to 4Gb so we can access it */
	if (!DPMI_setSelectorLimit(sregs.ds,0xFFFFFFFFUL))
		return NULL;

	/* Finally enable caching for the page tables that we just mapped in,
	 * since DOS4GW and PMODE/W create the page table entries without
	 * caching enabled which hurts the performance of the linear framebuffer
	 * as it disables write combining on Pentium Pro and above processors.
	 *
	 * For those processors cache disabling is better handled through the
	 * MTRR registers anyway (we can write combine a region but disable
	 * caching) so that MMIO register regions do not screw up.
	 */
	if (isCached) {
#ifdef	DOS4GW
		if ((PDB = _PM_getPDB()) != 0 && DSBaseAddr == 0) {
			int 	startPDB,endPDB,iPDB,startPage,endPage,start,end,iPage;
			ulong 	pageTable,*pPageTable;
			if (!pPDB) {
				if (PDB >= 0x100000)
					pPDB = (ulong*)DPMI_mapPhysicalToLinear(PDB,0xFFF);
				else
					pPDB = (ulong*)PDB;
				}
			if (pPDB) {
				startPDB = (linAddr >> 22) & 0x3FF;
				startPage = (linAddr >> 12) & 0x3FF;
				endPDB = ((linAddr+limit) >> 22) & 0x3FF;
				endPage = ((linAddr+limit) >> 12) & 0x3FF;
				for (iPDB = startPDB; iPDB <= endPDB; iPDB++) {
					pageTable = pPDB[iPDB] & ~0xFFF;
					if (pageTable >= 0x100000)
						pPageTable = (ulong*)DPMI_mapPhysicalToLinear(pageTable,0xFFF);
					else
						pPageTable = (ulong*)pageTable;
					start = (iPDB == startPDB) ? startPage : 0;
					end = (iPDB == endPDB) ? endPage : 0x3FF;
					for (iPage = start; iPage <= end; iPage++)
						pPageTable[iPage] &= ~0x18;
					}
				}
			}
#endif
		}

	/* Now return the base address of the memory into the default DS */
	return (void*)(linAddr - DSBaseAddr);
}

#if	defined(PM386)

/* Some DOS extender implementations do not directly support calling a
 * real mode procedure from protected mode. However we can simulate what
 * we need temporarily hooking the INT 6Ah vector with a small real mode
 * stub that will call our real mode code for us.
 */

static uchar int6AHandler[] = {
	0x00,0x00,0x00,0x00,		/* 	__PMODE_callReal variable			*/
	0xFB,						/*	sti									*/
	0x2E,0xFF,0x1E,0x00,0x00,	/*  call	[cs:__PMODE_callReal]		*/
	0xCF,						/*	iretf								*/
	};
static uchar *crPtr = NULL;	/* Pointer to of int 6A handler			*/
static uint crRSeg,crROff;	/* Real mode seg:offset of handler		*/

void PMAPI PM_callRealMode(uint seg,uint off, RMREGS *in,
	RMSREGS *sregs)
{
	uchar	*p;
	uint	oldSeg,oldOff;

	if (!crPtr) {
		/* Allocate and copy the memory block only once */
		crPtr = PM_allocRealSeg(sizeof(int6AHandler), &crRSeg, &crROff);
		memcpy(crPtr,int6AHandler,sizeof(int6AHandler));
		}
	PM_setWord(crPtr,off);				/* Plug in address to call	*/
	PM_setWord(crPtr+2,seg);
	p = PM_mapRealPointer(0,0x6A * 4);
	oldOff = PM_getWord(p);				/* Save old handler address	*/
	oldSeg = PM_getWord(p+2);
	PM_setWord(p,crROff+4);				/* Hook 6A handler			*/
	PM_setWord(p+2,crRSeg);
	PM_int86x(0x6A, in, in, sregs);		/* Call real mode code		*/
	PM_setWord(p,oldOff);				/* Restore old handler		*/
	PM_setWord(p+2,oldSeg);
}

#endif	/* PM386 */

/*-------------------------------------------------------------------------*/
/* DOS Real Mode support.												   */
/*-------------------------------------------------------------------------*/

#ifdef REALMODE

#ifndef	MK_FP
#define MK_FP(s,o)  ( (void far *)( ((ulong)(s) << 16) + \
					(ulong)(o) ))
#endif

void * PMAPI PM_mapRealPointer(uint r_seg,uint r_off)
{ return MK_FP(r_seg,r_off); }

void * PMAPI PM_getBIOSPointer(void)
{
	return MK_FP(0x40,0);
}

void * PMAPI PM_getA0000Pointer(void)
{
	return MK_FP(0xA000,0);
}

void * PMAPI PM_mapPhysicalAddr(ulong base,ulong limit,ibool isCached)
{
	uint sel = base >> 4;
	uint off = base & 0xF;
	limit = limit;
	return MK_FP(sel,off);
}

void PMAPI PM_freePhysicalAddr(void *ptr,ulong limit)
{ ptr = ptr; }

ulong PMAPI PM_getPhysicalAddr(void *p)
{ return 0xFFFFFFFFUL; }

void * PMAPI PM_mapToProcess(void *base,ulong limit)
{ return (void*)base; }

void * PMAPI PM_allocRealSeg(uint size,uint *r_seg,uint *r_off)
{
	/* Call malloc() to allocate the memory for us */
	void *p = malloc(size);
	*r_seg = FP_SEG(p);
	*r_off = FP_OFF(p);
	return p;
}

void PMAPI PM_freeRealSeg(void *mem)
{
	free(mem);
}

int PMAPI PM_int86(int intno, RMREGS *in, RMREGS *out)
{
	return PM_int386(intno,in,out);
}

int PMAPI PM_int86x(int intno, RMREGS *in, RMREGS *out,
	RMSREGS *sregs)
{
	return PM_int386x(intno,in,out,sregs);
}

void PMAPI PM_availableMemory(ulong *physical,ulong *total)
{
	PMREGS regs;

	regs.h.ah = 0x48;
	regs.x.bx = 0xFFFF;
	PM_int86(0x21,&regs,&regs);
	*physical = *total = regs.x.bx * 16UL;
}

ibool PMAPI PM_enableWriteCombine(ulong base,ulong length,uint type)
{ return false; }

#endif

/*-------------------------------------------------------------------------*/
/* Phar Lap TNT DOS Extender support.									   */
/*-------------------------------------------------------------------------*/

#ifdef TNT

#include <pldos32.h>
#include <pharlap.h>
#include <hw386.h>

static uchar *zeroPtr = NULL;

void * PMAPI PM_getBIOSPointer(void)
{
	if (!zeroPtr)
		zeroPtr = PM_mapPhysicalAddr(0,0xFFFFF,true);
	return (void*)(zeroPtr + 0x400);
}

void * PMAPI PM_getA0000Pointer(void)
{
	static void *bankPtr;
	if (!bankPtr)
		bankPtr = PM_mapPhysicalAddr(0xA0000,0xFFFF,true);
	return bankPtr;
}

void * PMAPI PM_mapPhysicalAddr(ulong base,ulong limit,ibool isCached)
{
	CONFIG_INF	config;
	ULONG		offset;
	int			err;
	ulong		baseAddr,baseOfs,newLimit;

	/* Round the physical address to a 4Kb boundary and the limit to a
	 * 4Kb-1 boundary before passing the values to TNT. If we round the
	 * physical address, then we also add an extra offset into the address
	 * that we return.
	 */
	baseOfs = base & 4095;
	baseAddr = base & ~4095;
	newLimit = ((limit+baseOfs+1+4095) & ~4095)-1;
	_dx_config_inf(&config, (UCHAR*)&config);
	err = _dx_map_phys(config.c_ds_sel,baseAddr,(newLimit + 4095) / 4096,&offset);
	if (err == 130) {
		/* If the TNT function failed, we are running in a DPMI environment
		 * and this function does not work. However we know how to handle
		 * DPMI properly, so we use our generic DPMI functions to do
		 * what the TNT runtime libraries can't.
		 */
		return DPMI_mapPhysicalAddr(base,limit,isCached);
		}
	if (err == 0)
		return (void*)(offset + baseOfs);
	return NULL;
}

void PMAPI PM_freePhysicalAddr(void *ptr,ulong limit)
{
	/* TODO: We should be able to free non-DPMI mappings with PharLap */
}

ulong PMAPI PM_getPhysicalAddr(void *p)
{ return 0xFFFFFFFFUL; }

void * PMAPI PM_mapToProcess(void *base,ulong limit)
{ return (void*)base; }

void * PMAPI PM_mapRealPointer(uint r_seg,uint r_off)
{
	if (!zeroPtr)
		zeroPtr = PM_mapPhysicalAddr(0,0xFFFFF);
	return (void*)(zeroPtr + MK_PHYS(r_seg,r_off));
}

void * PMAPI PM_allocRealSeg(uint size,uint *r_seg,uint *r_off)
{
	USHORT	addr,t;
	void	*p;

	if (_dx_real_alloc((size + 0xF) >> 4,&addr,&t) != 0)
		return 0;
	*r_seg = addr;					/* Real mode segment address	*/
	*r_off = 0;						/* Real mode segment offset		*/
	p = PM_mapRealPointer(*r_seg,*r_off);
	_PM_addRealModeBlock(p,addr);
	return p;
}

void PMAPI PM_freeRealSeg(void *mem)
{
	_dx_real_free(_PM_findRealModeBlock(mem));
}

#define INDPMI(reg)     rmregs.reg = regs->reg
#define OUTDPMI(reg)    regs->reg = rmregs.reg

void PMAPI DPMI_int86(int intno, DPMI_regs *regs)
{
	SWI_REGS	rmregs;

	memset(&rmregs, 0, sizeof(rmregs));
	INDPMI(eax); INDPMI(ebx); INDPMI(ecx); INDPMI(edx); INDPMI(esi); INDPMI(edi);

	_dx_real_int(intno,&rmregs);

	OUTDPMI(eax); OUTDPMI(ebx); OUTDPMI(ecx); OUTDPMI(edx); OUTDPMI(esi); OUTDPMI(edi);
	regs->flags = rmregs.flags;
}

#define IN(reg)     rmregs.reg = in->e.reg
#define OUT(reg)    out->e.reg = rmregs.reg

int PMAPI PM_int86(int intno, RMREGS *in, RMREGS *out)
{
	SWI_REGS	rmregs;

	memset(&rmregs, 0, sizeof(rmregs));
	IN(eax); IN(ebx); IN(ecx); IN(edx); IN(esi); IN(edi);

	_dx_real_int(intno,&rmregs);

	OUT(eax); OUT(ebx); OUT(ecx); OUT(edx); OUT(esi); OUT(edi);
	out->x.cflag = rmregs.flags & 0x1;
	return out->x.ax;
}

int PMAPI PM_int86x(int intno, RMREGS *in, RMREGS *out,
	RMSREGS *sregs)
{
	SWI_REGS	rmregs;

	memset(&rmregs, 0, sizeof(rmregs));
	IN(eax); IN(ebx); IN(ecx); IN(edx); IN(esi); IN(edi);
	rmregs.es = sregs->es;
	rmregs.ds = sregs->ds;

	_dx_real_int(intno,&rmregs);

	OUT(eax); OUT(ebx); OUT(ecx); OUT(edx); OUT(esi); OUT(edi);
	sregs->es = rmregs.es;
	sregs->cs = rmregs.cs;
	sregs->ss = rmregs.ss;
	sregs->ds = rmregs.ds;
	out->x.cflag = rmregs.flags & 0x1;
	return out->x.ax;
}

void PMAPI PM_availableMemory(ulong *physical,ulong *total)
{
	PMREGS 	r;
	uint	data[25];

	r.x.ax = 0x2520;				/* Get free memory info */
	r.x.bx = 0;
	r.e.edx = (uint)data;
	PM_int386(0x21, &r, &r);
	*physical = data[21] * 4096;
	*total = data[23] * 4096;
}

void *	PMAPI PM_allocLockedMem(uint size,ulong *physAddr)
{
	// Not implemented yet.
	return NULL;
}

void PMAPI PM_freeLockedMem(void *p,uint size)
{
}

#endif

/*-------------------------------------------------------------------------*/
/* Symantec C++ DOSX and FlashTek X-32/X-32VM support					   */
/*-------------------------------------------------------------------------*/

#if	defined(DOSX) || defined(X32VM)

#ifdef	X32VM
#include <x32.h>

#define	_x386_mk_protected_ptr(p)	_x32_mk_protected_ptr((void*)p)
#define	_x386_free_protected_ptr(p)	_x32_free_protected_ptr(p)
#define	_x386_zero_base_ptr			_x32_zero_base_ptr
#else
extern void *_x386_zero_base_ptr;
#endif

void * PMAPI PM_mapRealPointer(uint r_seg,uint r_off)
{
	return (void*)((ulong)_x386_zero_base_ptr + MK_PHYS(r_seg,r_off));
}

void * PMAPI PM_allocRealSeg(uint size,uint *r_seg,uint *r_off)
{
	PMREGS	r;

	r.h.ah = 0x48;					/* DOS function 48h - allocate mem	*/
	r.x.bx = (size + 0xF) >> 4;		/* Number of paragraphs to allocate	*/
	PM_int386(0x21, &r, &r);		/* Call DOS extender				*/
	if (r.x.cflag)
		return 0;					/* Could not allocate the memory	*/
	*r_seg = r.e.eax;
	*r_off = 0;
	return PM_mapRealPointer(*r_seg,*r_off);
}

void PMAPI PM_freeRealSeg(void *mem)
{
	/* Cannot de-allocate this memory */
	mem = mem;
}

#pragma pack(1)

typedef struct {
	ushort	intno;
	ushort	ds;
	ushort	es;
	ushort	fs;
	ushort	gs;
	ulong	eax;
	ulong	edx;
	} _RMREGS;

#pragma pack()

#define IN(reg)     regs.e.reg = in->e.reg
#define OUT(reg)    out->e.reg = regs.e.reg

int PMAPI PM_int86(int intno, RMREGS *in, RMREGS *out)
{
	_RMREGS rmregs;
	PMREGS  regs;
	PMSREGS	pmsregs;

	rmregs.intno = intno;
	rmregs.eax = in->e.eax;
	rmregs.edx = in->e.edx;
	IN(ebx); IN(ecx); IN(esi); IN(edi);
	regs.x.ax = 0x2511;
	regs.e.edx = (uint)(&rmregs);
	PM_segread(&pmsregs);
	PM_int386x(0x21,&regs,&regs,&pmsregs);

	OUT(eax); OUT(ebx); OUT(ecx); OUT(esi); OUT(edi);
	out->x.dx = rmregs.edx;
	out->x.cflag = regs.x.cflag;
	return out->x.ax;
}

int PMAPI PM_int86x(int intno, RMREGS *in, RMREGS *out, RMSREGS *sregs)
{
	_RMREGS rmregs;
	PMREGS  regs;
	PMSREGS	pmsregs;

	rmregs.intno = intno;
	rmregs.eax = in->e.eax;
	rmregs.edx = in->e.edx;
	rmregs.es = sregs->es;
	rmregs.ds = sregs->ds;
	IN(ebx); IN(ecx); IN(esi); IN(edi);
	regs.x.ax = 0x2511;
	regs.e.edx = (uint)(&rmregs);
	PM_segread(&pmsregs);
	PM_int386x(0x21,&regs,&regs,&pmsregs);

	OUT(eax); OUT(ebx); OUT(ecx); OUT(esi); OUT(edi);
	sregs->es = rmregs.es;
	sregs->ds = rmregs.ds;
	out->x.dx = rmregs.edx;
	out->x.cflag = regs.x.cflag;
	return out->x.ax;
}

void * PMAPI PM_getBIOSPointer(void)
{
	return (void*)((ulong)_x386_zero_base_ptr + 0x400);
}

void * PMAPI PM_getA0000Pointer(void)
{
	return (void*)((ulong)_x386_zero_base_ptr + 0xA0000);
}

void * PMAPI PM_mapPhysicalAddr(ulong base,ulong limit,ibool isCached)
{
	if (base > 0x100000)
		return _x386_map_physical_address((void*)base,limit);
	return (void*)((ulong)_x386_zero_base_ptr + base);
}

void PMAPI PM_freePhysicalAddr(void *ptr,ulong limit)
{
	/* Mapping cannot be freed */
}

ulong PMAPI PM_getPhysicalAddr(void *p)
{ return 0xFFFFFFFFUL; }

void * PMAPI PM_mapToProcess(void *base,ulong limit)
{ return (void*)base; }

ulong _cdecl _X32_getPhysMem(void);

void PMAPI PM_availableMemory(ulong *physical,ulong *total)
{
	PMREGS	regs;

	/* Get total memory available, including virtual memory */
	regs.x.ax = 0x350B;
	PM_int386(0x21,&regs,&regs);
	*total = regs.e.eax;

	/* Get physical memory available */
	*physical = _X32_getPhysMem();
	if (*physical > *total)
		*physical = *total;
}

void *	PMAPI PM_allocLockedMem(uint size,ulong *physAddr)
{
	// Not implemented yet.
	return NULL;
}

void PMAPI PM_freeLockedMem(void *p,uint size)
{
}

#endif

/*-------------------------------------------------------------------------*/
/* Borland's DPMI32, Watcom DOS4GW and DJGPP DPMI support routines		   */
/*-------------------------------------------------------------------------*/

#if	defined(DPMI32) || defined(DOS4GW) || defined(DJGPP)

void * PMAPI PM_getBIOSPointer(void)
{
	return PM_mapPhysicalAddr(0x400,0xFFFF,true);
}

void * PMAPI PM_getA0000Pointer(void)
{
	return PM_mapPhysicalAddr(0xA0000,0xFFFF,true);
}

void * PMAPI PM_mapPhysicalAddr(ulong base,ulong limit,ibool isCached)
{
#ifdef	DJGPP
	/* Enable near pointers for DJGPP V2 */
	__djgpp_nearptr_enable();
#endif
	return DPMI_mapPhysicalAddr(base,limit,isCached);
}

void PMAPI PM_freePhysicalAddr(void *ptr,ulong limit)
{
	/* Mapping cannot be freed */
	(void)ptr;
	(void)limit;
}

ulong PMAPI PM_getPhysicalAddr(void *p)
{
	/* Not implemented yet */
	(void)p;
	return 0xFFFFFFFFUL;
}

void * PMAPI PM_mapToProcess(void *base,ulong limit)
{
	(void)limit;
	return (void*)base;
}

void * PMAPI PM_mapRealPointer(uint r_seg,uint r_off)
{
	static uchar *zeroPtr = NULL;

	if (!zeroPtr)
		zeroPtr = PM_mapPhysicalAddr(0,0xFFFFF,true);
	return (void*)(zeroPtr + MK_PHYS(r_seg,r_off));
}

void * PMAPI PM_allocRealSeg(uint size,uint *r_seg,uint *r_off)
{
	PMREGS		r;
	void		*p;

	r.x.ax = 0x100;					/* DPMI allocate DOS memory			*/
	r.x.bx = (size + 0xF) >> 4;		/* number of paragraphs 			*/
	PM_int386(0x31, &r, &r);
	if (r.x.cflag)
		return NULL;				/* DPMI call failed					*/
	*r_seg = r.x.ax;				/* Real mode segment				*/
	*r_off = 0;
	p = PM_mapRealPointer(*r_seg,*r_off);
	_PM_addRealModeBlock(p,r.x.dx);
	return p;
}

void PMAPI PM_freeRealSeg(void *mem)
{
	PMREGS	r;

	r.x.ax = 0x101;						/* DPMI free DOS memory			*/
	r.x.dx = _PM_findRealModeBlock(mem);/* DX := selector from 0x100	*/
	PM_int386(0x31, &r, &r);
}

static DPMI_handler_t	DPMI_int10 = NULL;

void PMAPI DPMI_setInt10Handler(DPMI_handler_t handler)
{
	DPMI_int10 = handler;
}

void PMAPI DPMI_int86(int intno, DPMI_regs *regs)
{
	PMREGS		r;
	PMSREGS		sr;

	if (intno == 0x10 && DPMI_int10) {
		if (DPMI_int10(regs))
			return;
		}
	PM_segread(&sr);
	r.x.ax = 0x300;					/* DPMI issue real interrupt	*/
	r.h.bl = intno;
	r.h.bh = 0;
	r.x.cx = 0;
	sr.es = sr.ds;
	r.e.edi = (uint)regs;
	PM_int386x(0x31, &r, &r, &sr);	/* Issue the interrupt			*/
}

#define IN(reg)     rmregs.reg = in->e.reg
#define OUT(reg)    out->e.reg = rmregs.reg

int PMAPI PM_int86(int intno, RMREGS *in, RMREGS *out)
{
	DPMI_regs	rmregs;

	memset(&rmregs, 0, sizeof(rmregs));
	IN(eax); IN(ebx); IN(ecx); IN(edx); IN(esi); IN(edi);

	DPMI_int86(intno,&rmregs);		/* DPMI issue real interrupt	*/

	OUT(eax); OUT(ebx); OUT(ecx); OUT(edx); OUT(esi); OUT(edi);
	out->x.cflag = rmregs.flags & 0x1;
	return out->x.ax;
}

int PMAPI PM_int86x(int intno, RMREGS *in, RMREGS *out,
	RMSREGS *sregs)
{
	DPMI_regs	rmregs;

	memset(&rmregs, 0, sizeof(rmregs));
	IN(eax); IN(ebx); IN(ecx); IN(edx); IN(esi); IN(edi);
	rmregs.es = sregs->es;
	rmregs.ds = sregs->ds;

	DPMI_int86(intno,&rmregs);		/* DPMI issue real interrupt	*/

	OUT(eax); OUT(ebx); OUT(ecx); OUT(edx); OUT(esi); OUT(edi);
	sregs->es = rmregs.es;
	sregs->cs = rmregs.cs;
	sregs->ss = rmregs.ss;
	sregs->ds = rmregs.ds;
	out->x.cflag = rmregs.flags & 0x1;
	return out->x.ax;
}

#pragma pack(1)

typedef struct {
		uint	LargestBlockAvail;
		uint	MaxUnlockedPage;
		uint	LargestLockablePage;
		uint    LinAddrSpace;
		uint    NumFreePagesAvail;
		uint    NumPhysicalPagesFree;
		uint    TotalPhysicalPages;
		uint	FreeLinAddrSpace;
		uint	SizeOfPageFile;
		uint	res[3];
		} MemInfo;

#pragma pack()

void PMAPI PM_availableMemory(ulong *physical,ulong *total)
{
	PMREGS 	r;
	PMSREGS	sr;
	MemInfo	memInfo;

	PM_segread(&sr);
	r.x.ax = 0x500;					/* DPMI get free memory info */
	sr.es = sr.ds;
	r.e.edi = (uint)&memInfo;
	PM_int386x(0x31, &r, &r, &sr);	/* Issue the interrupt */
	*physical = memInfo.NumPhysicalPagesFree * 4096;
	*total = memInfo.LargestBlockAvail;
	if (*total < *physical)
		*physical = *total;
}

void *	PMAPI PM_allocLockedMem(uint size,ulong *physAddr)
{
	/* Not implemented yet */
	(void)size;
	(void)physAddr;
	return NULL;
}

void PMAPI PM_freeLockedMem(void *p,uint size)
{
	(void)p;
	(void)size;
}

#endif

#ifndef	__16BIT__

/****************************************************************************
REMARKS:
Call the VBE/Core software interrupt to change display banks.
****************************************************************************/
void PMAPI PM_setBankA(
	int bank)
{
	DPMI_regs	regs;
	memset(&regs, 0, sizeof(regs));
	regs.eax = 0x4F05;
	regs.ebx = 0x0000;
	regs.edx = bank;
	DPMI_int86(0x10,&regs);
}

/****************************************************************************
REMARKS:
Call the VBE/Core software interrupt to change display banks.
****************************************************************************/
void PMAPI PM_setBankAB(
	int bank)
{
	DPMI_regs	regs;
	memset(&regs, 0, sizeof(regs));
	regs.eax = 0x4F05;
	regs.ebx = 0x0000;
	regs.edx = bank;
	DPMI_int86(0x10,&regs);
	regs.eax = 0x4F05;
	regs.ebx = 0x0001;
	regs.edx = bank;
	DPMI_int86(0x10,&regs);
}

/****************************************************************************
REMARKS:
Call the VBE/Core software interrupt to change display start address.
****************************************************************************/
void PMAPI PM_setCRTStart(
	int x,
	int y,
	int waitVRT)
{
	DPMI_regs	regs;
	memset(&regs, 0, sizeof(regs));
	regs.eax = 0x4F07;
	regs.ebx = waitVRT;
	regs.ecx = x;
	regs.edx = y;
	DPMI_int86(0x10,&regs);
}

#endif
