/****************************************************************************
*
*					SciTech OS Portability Manager Library
*
*  ========================================================================
*
*    The contents of this file are subject to the SciTech MGL Public
*    License Version 1.0 (the "License"); you may not use this file
*    except in compliance with the License. You may obtain a copy of
*    the License at http://www.scitechsoft.com/mgl-license.txt
*
*    Software distributed under the License is distributed on an
*    "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
*    implied. See the License for the specific language governing
*    rights and limitations under the License.
*
*    The Original Code is Copyright (C) 1991-1998 SciTech Software, Inc.
*
*    The Initial Developer of the Original Code is SciTech Software, Inc.
*    All Rights Reserved.
*
*  ========================================================================
*
* Language:		ANSI C
* Environment:	Any
*
* Description:	Main module to implement the Zen Timer support functions.
*
****************************************************************************/

#include "ztimer.h"
#include "pmapi.h"
#include "oshdr.h"

/*----------------------------- Implementation ----------------------------*/

/* External Intel assembler functions */
#ifdef	__INTEL__
/* {secret} */
ibool	_ASMAPI _CPU_haveCPUID(void);
/* {secret} */
ibool	_ASMAPI _CPU_check80386(void);
/* {secret} */
ibool	_ASMAPI _CPU_check80486(void);
/* {secret} */
int 	_ASMAPI _CPU_inp(int port);
/* {secret} */
void 	_ASMAPI _CPU_outp(int port,int value);
/* {secret} */
uint	_ASMAPI _CPU_checkCPUID(void);
/* {secret} */
uint	_ASMAPI _CPU_getCPUIDModel(void);
/* {secret} */
uint	_ASMAPI _CPU_getCPUIDFeatures(void);
/* {secret} */
uint	_ASMAPI _CPU_have3DNow(void);
/* {secret} */
ibool	_ASMAPI _CPU_checkClone(void);
/* {secret} */
ulong	_ASMAPI _CPU_quickRDTSC(void);
/* {secret} */
void	_ASMAPI	_CPU_runBSFLoop(ulong iterations);
/* {secret} */
void ZTimerQuickInit(void);
#define	CPU_HaveMMX		0x00800000
#define	CPU_HaveRDTSC	0x00000010
#endif

#if		defined(__SMX32__)
#include "smx/cpuinfo.c"
#elif	defined(__RTTARGET__)
#include "rttarget/cpuinfo.c"
#elif	defined(__REALDOS__)
#include "dos/cpuinfo.c"
#elif	defined(__WIN32_VXD__)
#include "vxd/cpuinfo.c"
#elif	defined(__WINDOWS32__)
#include "win32/cpuinfo.c"
#elif	defined(__OS2__)
#include "os2/cpuinfo.c"
#elif	defined(__LINUX__)
#include "linux/cpuinfo.c"
#elif	defined(__QNX__)
#include "qnx/cpuinfo.c"
#elif	defined(__BEOS__)
#include "beos/cpuinfo.c"
#else
#error	CPU library not ported to this platform yet!
#endif

/*------------------------ Public interface routines ----------------------*/

/****************************************************************************
REMARKS:
Read an I/O port location.
****************************************************************************/
static uchar rdinx(
	int port,
	int index)
{
	_CPU_outp(port,index);
	return _CPU_inp(port+1);
}

/****************************************************************************
REMARKS:
Write an I/O port location.
****************************************************************************/
static void wrinx(
	ushort port,
	ushort index,
	ushort value)
{
	_CPU_outp(port,index);
	_CPU_outp(port+1,value);
}

/****************************************************************************
REMARKS:
Enables the Cyrix CPUID instruction to properly detect MediaGX and 6x86
processors.
****************************************************************************/
static void _CPU_enableCyrixCPUID(void)
{
	uchar	ccr3;

	PM_init();
	ccr3 = rdinx(0x22,0xC3);
	wrinx(0x22,0xC3,(uchar)(ccr3 | 0x10));
	wrinx(0x22,0xE8,(uchar)(rdinx(0x22,0xE8) | 0x80));
	wrinx(0x22,0xC3,ccr3);
}

/****************************************************************************
DESCRIPTION:
Returns the type of processor in the system.

HEADER:
ztimer.h

RETURNS:
Numerical identifier for the installed processor

REMARKS:
Returns the type of processor in the system. Note that if the CPU is an
unknown Pentium family processor that we don't have an enumeration for,
the return value will be greater than or equal to the value of CPU_UnkPentium
(depending on the value returned by the CPUID instruction).

SEE ALSO:
CPU_getProcessorSpeed, CPU_haveMMX
****************************************************************************/
uint ZAPI CPU_getProcessorType(void)
{
#if		defined(__INTEL__)
	uint			cpu,vendor,model;
	static ibool	firstTime = true;

	if (_CPU_haveCPUID()) {
		cpu = _CPU_checkCPUID();
		vendor = cpu & ~CPU_mask;
		if (vendor == CPU_Intel) {
			/* Check for Intel processors */
			switch (cpu & CPU_mask) {
				case 4:	cpu = CPU_i486;			break;
				case 5:	cpu = CPU_Pentium;		break;
				case 6:
					if ((model = _CPU_getCPUIDModel()) == 1)
						cpu = CPU_PentiumPro;
					else if (model <= 6)
						cpu = CPU_PentiumII;
					else if (model == 7)
						cpu = CPU_PentiumIII;
					break;
				default:
					cpu = CPU_UnkIntel;
				}
			}
		else if (vendor == CPU_Cyrix) {
			/* Check for Cyrix processors */
			switch (cpu & CPU_mask) {
				case 4:
					if ((model = _CPU_getCPUIDModel()) == 4)
						cpu = CPU_CyrixMediaGX;
					else
						cpu = CPU_UnkCyrix;
					break;
				case 5:
					if ((model = _CPU_getCPUIDModel()) == 2)
						cpu = CPU_Cyrix6x86;
					else if (model == 4)
						cpu = CPU_CyrixMediaGXm;
					else
						cpu = CPU_UnkCyrix;
					break;
				case 6:
					if ((model = _CPU_getCPUIDModel()) <= 1)
						cpu = CPU_Cyrix6x86MX;
					else
						cpu = CPU_UnkCyrix;
					break;
				default:
					cpu = CPU_UnkCyrix;
				}
			}
		else if (vendor == CPU_AMD) {
			/* Check for AMD processors */
			switch (cpu & CPU_mask) {
				case 4:
					if ((model = _CPU_getCPUIDModel()) == 0)
						cpu = CPU_AMDAm5x86;
					else
						cpu = CPU_AMDAm486;
					break;
				case 5:
					if ((model = _CPU_getCPUIDModel()) <= 3)
						cpu = CPU_AMDK5;
					else if (model <= 7)
						cpu = CPU_AMDK6;
					else if (model == 8)
						cpu = CPU_AMDK6_2;
					else if (model == 9)
						cpu = CPU_AMDK6_III;
					else
						cpu = CPU_UnkAMD;
					break;
				case 6:
					if ((model = _CPU_getCPUIDModel()) == 1)
						cpu = CPU_AMDAthlon;
					break;
				default:
					cpu = CPU_UnkAMD;
				}
			}
		else if (vendor == CPU_IDT) {
			/* Check for IDT WinChip processors */
			switch (cpu & CPU_mask) {
				case 5:
					if ((model = _CPU_getCPUIDModel()) <= 4)
						cpu = CPU_WinChipC6;
					else if (model == 8)
						cpu = CPU_WinChip2;
					else
						cpu = CPU_UnkIDT;
					break;
				default:
					cpu = CPU_UnkIDT;
				}
			}
		else {
			/* Assume a Pentium compatible Intel clone */
			cpu = CPU_Pentium;
			}
		return cpu | vendor;
		}
	else {
		if (_CPU_check80386())
			cpu = CPU_i386;
		else  if (_CPU_check80486()) {
			/* If we get here we may have a Cyrix processor so we can try
			 * enabling the CPUID instruction and trying again.
			 */
			if (firstTime) {
				firstTime = false;
				_CPU_enableCyrixCPUID();
				return CPU_getProcessorType();
				}
			cpu = CPU_i486;
			}
		else
			cpu = CPU_Pentium;
		if (!_CPU_checkClone())
			return cpu | CPU_Intel;
		return cpu;
		}
#elif	defined(__ALPHA__)
	return CPU_Alpha;
#elif	defined(__MIPS__)
	return CPU_Mips;
#elif	defined(__PPC__)
	return CPU_PowerPC;
#endif
}

/****************************************************************************
DESCRIPTION:
Returns true if the processor supports Intel MMX extensions.

HEADER:
ztimer.h

RETURNS:
True if MMX is available, false if not.

REMARKS:
This function determines if the processor supports the Intel MMX extended
instruction set.

SEE ALSO:
CPU_getProcessorType, CPU_getProcessorSpeed, CPU_have3DNow, CPU_haveKNI
****************************************************************************/
ibool ZAPI CPU_haveMMX(void)
{
#ifdef	__INTEL__
	if (_CPU_haveCPUID())
		return (_CPU_getCPUIDFeatures() & CPU_HaveMMX) != 0;
	return false;
#else
	return false;
#endif
}

/****************************************************************************
DESCRIPTION:
Returns true if the processor supports AMD 3DNow! extensions.

HEADER:
ztimer.h

RETURNS:
True if 3DNow! is available, false if not.

REMARKS:
This function determines if the processor supports the AMD 3DNow! extended
instruction set.

SEE ALSO:
CPU_getProcessorType, CPU_getProcessorSpeed, CPU_haveMMX, CPU_haveKNI
****************************************************************************/
ibool ZAPI CPU_have3DNow(void)
{
#ifdef	__INTEL__
	if (_CPU_haveCPUID())
		return _CPU_have3DNow();
	return false;
#else
	return false;
#endif
}

/****************************************************************************
DESCRIPTION:
Returns true if the processor supports Intel KNI extensions.

HEADER:
ztimer.h

RETURNS:
True if Intel KNI is available, false if not.

REMARKS:
This function determines if the processor supports the Intel KNI extended
instruction set.

SEE ALSO:
CPU_getProcessorType, CPU_getProcessorSpeed, CPU_haveMMX, CPU_have3DNow
****************************************************************************/
ibool ZAPI CPU_haveKNI(void)
{
#ifdef	__INTEL__
	/* TODO: This needs to be implemented for the Pentium III */
	return false;
#else
	return false;
#endif
}

/****************************************************************************
RETURNS:
True if the RTSC instruction is available, false if not.

REMARKS:
This function determines if the processor supports the Intel RDTSC
instruction, for high precision timing. If the processor is not an Intel or
Intel clone CPU, this function will always return false.

DESCRIPTION:
Returns true if the processor supports RDTSC extensions.

HEADER:
ztimer.h

RETURNS:
True if RTSC is available, false if not.

REMARKS:
This function determines if the processor supports the RDTSC instruction
for reading the processor time stamp counter.

SEE ALSO:
CPU_getProcessorType, CPU_getProcessorSpeed, CPU_haveMMX, CPU_have3DNow
****************************************************************************/
ibool ZAPI CPU_haveRDTSC(void)
{
#ifdef	__INTEL__
	if (_CPU_haveCPUID())
		return (_CPU_getCPUIDFeatures() & CPU_HaveRDTSC) != 0;
	return false;
#else
	return false;
#endif
}

#ifdef	__INTEL__

#define ITERATIONS		16000
#define	SAMPLINGS		2
#define	INNER_LOOPS		500

/****************************************************************************
REMARKS:
If processor does not support time stamp reading, but is at least a 386 or
above, utilize method of timing a loop of BSF instructions which take a
known number of cycles to run on i386(tm), i486(tm), and Pentium(R)
processors.
****************************************************************************/
static ulong GetBSFCpuSpeed(
	ulong cycles)
{
	CPU_largeInteger t0,t1,count_freq;
	ulong	ticks;				/* Microseconds elapsed during test		*/
	ulong 	current;      		/* Variable to store time elapsed		*/
	int 	i,j,iPriority;
	ulong 	lowest  = (ulong)-1;

	iPriority = SetMaxThreadPriority();
	GetCounterFrequency(&count_freq);
	for (i = 0; i < SAMPLINGS; i++) {
		GetCounter(&t0);
		for (j = 0; j < INNER_LOOPS; j++)
			_CPU_runBSFLoop(ITERATIONS);
		GetCounter(&t1);
		current = t1.low - t0.low;
		if (current < lowest)
			lowest = current;
		}
	RestoreThreadPriority(iPriority);

	/* Compute frequency */
	ticks = (ulong)((lowest * 1000000.0) / (float)(count_freq.low));
	if ((ticks % count_freq.low) > (count_freq.low/2))
		ticks++;			/* Round up if necessary */
	if (ticks == 0)
		return 0;
	return ((cycles*INNER_LOOPS)/ticks);
}

/****************************************************************************
REMARKS:
Utility function to return the absolute value of a number
****************************************************************************/
static long _abs(long a)
{
	return (a >= 0 ? a : -a);
}

#define TOLERANCE		1

/****************************************************************************
REMARKS:
On processors supporting the Read Time Stamp opcode, compare elapsed
time on the High-Resolution Counter with elapsed cycles on the Time
Stamp Register.

The inner loop runs up to 20 times oruntil the average of the previous
three calculated frequencies is within 1 MHz of each of the individual
calculated frequencies. This resampling increases the accuracy of the
results since outside factors could affect this calculation.
****************************************************************************/
static ulong GetRDTSCCpuSpeed(void)
{
	CPU_largeInteger	t0,t1,count_freq;
	ulong 	freq=0;					/* Most current frequ. calculation                      */
	ulong 	freq2=0;				/* 2nd most current frequ. calc.                        */
	ulong	freq3;					/* 3rd most current frequ. calc.                        */
	ulong 	total;					/* Sum of previous three frequency calculations         */
	int    	tries=0;				/* Number of times a calculation has been made          */
	ulong  	total_cycles=0, cycles;	/* Clock cycles elapsed during test                     */
	ulong  	stamp0, stamp1;			/* Time Stamp Variable for beginning and end of test    */
	ulong  	total_ticks=0, ticks;	/* Microseconds elapsed during test                     */
	int		iPriority;

	iPriority = SetMaxThreadPriority();
	GetCounterFrequency(&count_freq);
	do {
		tries++;		/* Increment number of times sampled */
		freq3 = freq2;	/* Shift frequencies back */
		freq2 = freq;

		/* Loop until 100 ticks have passed since last read of hi-res
		 * counter. This accounts for overhead later.
		 */
		GetCounter(&t0);
		t1.low = t0.low;
		t1.high = t0.high;
		while ((t1.low - t0.low) < 100) {
			GetCounter(&t1);
			stamp0 = _CPU_quickRDTSC();
			}

		/* Loop until 30000 ticks have passed since last read of hi-res counter.
		 * This allows for elapsed time for sampling. For a hi-res frequency
		 * of 1Mhz, this is about 0.03 of a second. The frequency reported
		 * by the OS dependent code should be tuned to provide a good
		 * sample period depending on the accuracy of the OS timers (ie:
		 * if the accuracy is lower, lower the frequency to spend more time
		 * in the inner loop to get better accuracy).
		 */
		t0.low = t1.low;
		t0.high = t1.high;
		while ((t1.low - t0.low) < 30000) {
			GetCounter(&t1);
			stamp1 = _CPU_quickRDTSC();
			}

		/* Find the difference during the timing loop */
		cycles = stamp1 - stamp0;
		ticks = t1.low - t0.low;

		/* Compute the CPU frequency */
		ticks = (ulong)((ticks * 1000000.0) / count_freq.low);
		total_ticks += ticks;
		total_cycles += cycles;
		if ((ticks % count_freq.low) > (count_freq.low/2))
			ticks++;			/* Round up if necessary 					*/
		freq = cycles/ticks;	/* Cycles / us  = MHz 						*/
		if ((cycles % ticks) > (ticks/2))
			freq++;				/* Round up if necessary					*/
		total = (freq + freq2 + freq3);
								/* Total last three frequency calculations	*/
		} while ( (tries < 3 ) ||
				  ((tries < 20) &&
				   ((_abs(3 * freq -total) > (3*TOLERANCE)) ||
					(_abs(3 * freq2-total) > (3*TOLERANCE)) ||
					(_abs(3 * freq3-total) > (3*TOLERANCE)))));
	RestoreThreadPriority(iPriority);
	return (total_cycles / total_ticks);
}

#endif	/* __INTEL__ */

/****************************************************************************
DESCRIPTION:
Returns the speed of the processor in Mhz.

HEADER:
ztimer.h

RETURNS:
Processor speed in Mhz.

REMARKS:
This function returns the speed of the CPU in Mhz. Note that if the speed
cannot be determined, this function will return 0.

SEE ALSO:
CPU_getProcessorType, CPU_haveMMX
****************************************************************************/
ulong ZAPI CPU_getProcessorSpeed(void)
{
#if defined(__INTEL__)
	/* Number of cycles needed to execute a single BSF instruction on i386+
	 * processors.
	 */
	ulong	cpuSpeed;
	uint	i;
	static 	ulong intel_cycles[] = {
		115,47,43,
		};
	static 	ulong cyrix_cycles[] = {
		38,38,52,52,
		};
	static 	ulong known_speeds[] = {
		650,600,550,500,450,400,350,333,300,266,233,200,166,150,133,120,100,90,75,66,60,50,33,20,0,
		};
	if (CPU_haveRDTSC()) {
		cpuSpeed = GetRDTSCCpuSpeed();
		}
	else {
		int processor = CPU_getProcessorType() & CPU_mask;
		uint vendor = CPU_getProcessorType() & ~CPU_mask;
		if (vendor == CPU_Intel)
			cpuSpeed = GetBSFCpuSpeed(ITERATIONS * intel_cycles[processor - CPU_i386]);
		else if (vendor == CPU_Cyrix)
			cpuSpeed = GetBSFCpuSpeed(ITERATIONS * cyrix_cycles[processor - CPU_Cyrix6x86]);
		else
			return 0;
		}

	/* Now normalise the results given known processors speeds, if the
	 * speed we measure is within 2Mhz of the expected values
	 */
	for (i = 0; known_speeds[i] != 0; i++) {
		if (cpuSpeed >= (known_speeds[i]-2) && cpuSpeed <= (known_speeds[i]+2)) {
			return known_speeds[i];
			}
		}
	return cpuSpeed;
#else
	return 0;
#endif
}
