#include <linux/module.h>

#include <linux/kernel.h> /* printk() */
#include <linux/malloc.h> /* kmalloc() */
#include <linux/fs.h>     /* everything... */
#include <linux/errno.h>  /* error codes */
#include <linux/types.h>  /* size_t */
#include <linux/proc_fs.h>
#include <linux/fcntl.h>        /* O_ACCMODE */
#include <linux/ioport.h>
#include <linux/pci.h>

#include <asm/uaccess.h>
#include <asm/system.h>   /* cli(), *_flags */
#include <asm/segment.h>  /* memcpy and such */
#include <asm/io.h>

#include "svgalib_helper.h"

int num_devices=0;

static struct sh_pci_device 
       *sh_pci_devs[SVGALIB_HELPER_MAX_NR_DEVICES];

#if 0
static void check_io_range(port,device) {
       
       if((port&0xffe0)==0x3c0)
           printk(KERN_DEBUG"svgalib_helper: io to port 0x%x\n",port);
}
#else
#define check_io_range(port,device)
#endif

static int svgalib_helper_ioctl( struct inode *inode, struct file *filp, 
                          unsigned int cmd, unsigned long arg) {

    io_t iov;
    int minor = MINOR(inode->i_rdev);

    if(_IOC_TYPE(cmd)!=SVGALIB_HELPER_IOC_MAGIC) return -EINVAL;
    switch(_IOC_NR(cmd)) {
        case _IOC_NR(SVGALIB_HELPER_IOCSOUTB):
           copy_from_user(&iov,(char *)arg,sizeof(iov));
           check_io_range(iov.port,minor);
           outb(iov.val,iov.port);
        break;
        case _IOC_NR(SVGALIB_HELPER_IOCSOUTW):
           copy_from_user(&iov,(char *)arg,sizeof(iov));
           check_io_range(iov.port,minor);
           outw(iov.val,iov.port);
        break;
        case _IOC_NR(SVGALIB_HELPER_IOCSOUTL):
           copy_from_user(&iov,(char *)arg,sizeof(iov));
           check_io_range(iov.port,minor);
           outl(iov.val,iov.port);
        break;
        case _IOC_NR(SVGALIB_HELPER_IOCGINB):
           copy_from_user(&iov,(char *)arg,sizeof(iov));
           check_io_range(iov.port,minor);
           iov.val=inb(iov.port);
           copy_to_user((char *)arg,&iov,sizeof(iov));
        break;
        case _IOC_NR(SVGALIB_HELPER_IOCGINW):
           copy_from_user(&iov,(char *)arg,sizeof(iov));
           check_io_range(iov.port,minor);
           iov.val=inw(iov.port);
           copy_to_user((char *)arg,&iov,sizeof(iov));
        break;
        case _IOC_NR(SVGALIB_HELPER_IOCGINL):
           copy_from_user(&iov,(char *)arg,sizeof(iov));
           check_io_range(iov.port,minor);
           iov.val=inl(iov.port);
           copy_to_user((char *)arg,&iov,sizeof(iov));
        break;
        case _IOC_NR(SVGALIB_HELPER_IOCDVMA): {
           struct vm_area_struct *vma;  
           windowing_t winp;
           copy_from_user(&winp,(char *)arg,sizeof(winp));
           
           vma=current->mm->mmap;
           while(vma) {
              printk(KERN_DEBUG "svgalib_helper: vma from %.8lx to %.8lx\n",
                 vma->vm_start, vma->vm_end);
              vma=vma->vm_next;
           }
        }
        break;
        case _IOC_NR(SVGALIB_HELPER_IOCSWIND): {
           struct vm_area_struct *vma;  
           
           vma=current->mm->mmap;
           while(vma) {
              printk(KERN_DEBUG "svgalib_helper: vma from %.8lx to %.8lx\n",
                 vma->vm_start, vma->vm_end);
              vma=vma->vm_next;
           }
        }
        break;
        case _IOC_NR(SVGALIB_HELPER_IOCIOPERM):
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,3,0)
           if (current->thread.ioperm!=offsetof(struct thread_struct, io_bitmap)) {
              current->thread.ioperm=offsetof(struct thread_struct, io_bitmap);
              memset(current->thread.io_bitmap,0xff,(IO_BITMAP_SIZE+1)*4);
           }
           if(arg) {
               current->thread.io_bitmap[0x1e]=0xffffffff;
           } else {
               current->thread.io_bitmap[0x1e]=0;           
           }           
#else
           if (current->tss.bitmap!=offsetof(struct thread_struct, io_bitmap)) {
              current->tss.bitmap=offsetof(struct thread_struct, io_bitmap);
              memset(current->tss.io_bitmap,0xff,(IO_BITMAP_SIZE+1)*4);
           }
           if(arg) {
               current->tss.io_bitmap[0x1e]=0xffffffff;
           } else {
               current->tss.io_bitmap[0x1e]=0;           
           }
#endif
        break;
        default: return -EINVAL;
    }
    return 0;
}   			


static int svgalib_helper_open( struct inode *inode, struct file * filp) {

   int minor = MINOR(inode->i_rdev);
   
   if(minor>num_devices) return -ENODEV;
   
   MOD_INC_USE_COUNT;
   
   return 0;
}

static int svgalib_helper_release( struct inode *inode, struct file *filp) {
   MOD_DEC_USE_COUNT;
   
   return 0;
}

static int svgalib_helper_mmap(struct file *filp, struct vm_area_struct *vma) {
   int start=vma->vm_start;
   int end=vma->vm_end;
   int minor = MINOR(filp->f_dentry->d_inode->i_rdev);
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,3,0)
   int ofs=vma->vm_pgoff*PAGE_SIZE;
#else
   int ofs=vma->vm_offset;
#endif
   int i,j;
   if(minor==0) {
       if(((ofs>=0xa0000)&&(ofs+(end-start)<=0x100000))||
          ((ofs==0)&&(start==0)&&(end==0x1000))) {
          if(ofs>=0xa0000)vma->vm_flags|=VM_IO;
          if (remap_page_range(start, ofs, end-start, 
                               vma->vm_page_prot)) return -EAGAIN;
          return 0;
       }
       for(i=0;i<num_devices;i++)
           for(j=0;j<6;j++) if(sh_pci_devs[i]->mem[j])
               if((ofs>=sh_pci_devs[i]->mem[j])&&
                   (ofs+(end-start)<=sh_pci_devs[i]->mem[j]+sh_pci_devs[i]->len[j])) {
                   vma->vm_flags|=VM_IO;
                   if (remap_page_range(start, ofs, end-start, 
                                        vma->vm_page_prot)) return -EAGAIN;
                   return 0;
               }
   } else if(minor<16) {
      	   i=minor;	
           for(j=0;j<6;j++) if(sh_pci_devs[i]->mem[j])
               if((ofs>=sh_pci_devs[i]->mem[j])&&
                   (ofs+(end-start)<=sh_pci_devs[i]->mem[j]+sh_pci_devs[i]->len[j])) {
                   vma->vm_flags|=VM_IO;
                   if (remap_page_range(start, ofs, end-start, 
                                        vma->vm_page_prot)) return -EAGAIN;
                   return 0;
               }
   }
   return -EAGAIN;
}

struct file_operations svgalib_helper_fops = {
   NULL,	/* llseek */
   NULL,	/* read */
   NULL,	/* write */
   NULL,	/* readdir */
   NULL,	/* poll */
   svgalib_helper_ioctl,	/* ioctl */
   svgalib_helper_mmap,	/* mmap */
   svgalib_helper_open,	/* open */
   NULL,	/* flush */
   svgalib_helper_release	/* release */
};

int init_module(void)
{
    int result, i;
    struct pci_dev *dev=NULL;

    /*
     * Register your major, and accept a dynamic number
     */
    result = register_chrdev(SVGALIB_HELPER_MAJOR, "svgalib_helper", &svgalib_helper_fops);
    if (result < 0) {
        printk(KERN_WARNING "svgalib_helper: can't get major %d\n",SVGALIB_HELPER_MAJOR);
        return result;
    }

    if((sh_pci_devs[0]=kmalloc(sizeof(struct sh_pci_device),GFP_KERNEL))==NULL) {
        goto nomem_error;
    }
    memset(sh_pci_devs[0],0,sizeof(struct sh_pci_device));
    num_devices=1;
    for(i=1;i<SVGALIB_HELPER_MAX_NR_DEVICES;i++) sh_pci_devs[i]=NULL;

    if(pci_present()) {
        while((dev=pci_find_class(PCI_CLASS_DISPLAY_VGA<<8,dev))) {
            if((sh_pci_devs[num_devices]=kmalloc(sizeof(struct sh_pci_device),GFP_KERNEL))==NULL) {
                goto nomem_error;
            }
            memset(sh_pci_devs[num_devices],0,sizeof(struct sh_pci_device));
            sh_pci_devs[num_devices]->dev=dev;
            pci_read_config_word(dev,0,&sh_pci_devs[num_devices]->vendor);
            pci_read_config_word(dev,2,&sh_pci_devs[num_devices]->id);
            pci_read_config_byte(dev,8,&sh_pci_devs[num_devices]->revision);
            printk(KERN_DEBUG "svgalib_helper: device%d: vendor:%.4x id:%.4x\n",num_devices,
            sh_pci_devs[num_devices]->vendor,sh_pci_devs[num_devices]->id);
            for(i=0;i<6;i++){
                unsigned int t; 
                int len;
                pci_read_config_dword(dev,16+4*i,&result);
                if(result) {
                    pci_read_config_dword(dev,16+4*i,&result);
                    pci_write_config_dword(dev,16+4*i,0xffffffff);
                    pci_read_config_dword(dev,16+4*i,&t);
                    pci_write_config_dword(dev,16+4*i,result);
                    len = ~(t&~0xf)+1;
                    if (len){
                       sh_pci_devs[num_devices]->mem[i]=result&~0xf;
                       sh_pci_devs[num_devices]->flags[i]=result&0xf;
                       sh_pci_devs[num_devices]->len[i]=len;
                       printk(KERN_DEBUG "device%d: region%d, base=%.8x len=%d\n",
                       num_devices,i,result&(~0xf),len);
                    }
                }
            }
            num_devices++;
        }
    }

    EXPORT_NO_SYMBOLS;

    return 0; /* succeed */

    nomem_error:
        for(i=0;i<SVGALIB_HELPER_MAX_NR_DEVICES;i++) 
            if(sh_pci_devs[i])kfree(sh_pci_devs[i]);     
    unregister_chrdev(SVGALIB_HELPER_MAJOR, "svgalib_helper");
    return result;
}

void cleanup_module(void)
{
    int i;
    for(i=0;i<SVGALIB_HELPER_MAX_NR_DEVICES;i++) 
        if(sh_pci_devs[i]) {
            kfree(sh_pci_devs[i]);
        }
     
    unregister_chrdev(SVGALIB_HELPER_MAJOR, "svgalib_helper");
}


