#include <linux/module.h>

#include <linux/kernel.h> /* printk() */
#include <linux/malloc.h> /* kmalloc() */
#include <linux/fs.h>     /* everything... */
#include <linux/errno.h>  /* error codes */
#include <linux/types.h>  /* size_t */
#include <linux/stddef.h>
#include <linux/proc_fs.h>
#include <linux/fcntl.h>        /* O_ACCMODE */
#include <linux/ioport.h>
#include <linux/pci.h>
#include <linux/devfs_fs_kernel.h>

#ifndef _LINUX_DEVFS_FS_KERNEL_H
static inline int devfs_register_chrdev (unsigned int major, const char *name,
                                         struct file_operations *fops)
{
    return register_chrdev (major, name, fops);
}
static inline int devfs_unregister_chrdev (unsigned int major,const char *name)
{
    return unregister_chrdev (major, name);
}
#endif

#include <asm/uaccess.h>
#include <asm/system.h>   /* cli(), *_flags */
#include <asm/segment.h>  /* memcpy and such */
#include <asm/io.h>
#include <asm/pgtable.h>

#include "svgalib_helper.h"

int num_devices=0;

static struct sh_pci_device 
       *sh_pci_devs[SVGALIB_HELPER_MAX_NR_DEVICES];

#ifdef CONFIG_DEVFS_FS
static devfs_handle_t devfs_handle;
#endif

#define check_io_range(port,device)

static struct pci_dev *get_pci_dev(int pcipos, int minor) {
    
    if(minor>=num_devices) return NULL;
    if(minor>0) {
        return sh_pci_devs[minor]->dev;
    } else {
        int i;
        int b, d;

        d=pcipos&0xff;
        b=(pcipos>>8)&0xff;
        for(i=1;i<num_devices;i++) {
            if((b==sh_pci_devs[i]->dev->bus->number) && (d==sh_pci_devs[i]->dev->devfn)) 
                return sh_pci_devs[i]->dev;
        }
        return NULL;
    }
}

static int svgalib_helper_ioctl( struct inode *inode, struct file *filp, 
                          unsigned int cmd, unsigned long arg) {

    io_t iov;
    pcic_t pciv;
    int minor = MINOR(inode->i_rdev);
    struct pci_dev *pdev;
    io_string_t iostr;
    int i;
    u8 pb;
    u16 pw;
    u32 pl;
    unsigned char *outb_str;

    if(_IOC_TYPE(cmd)!=SVGALIB_HELPER_IOC_MAGIC) {
        return -EINVAL;
    }

    switch(_IOC_NR(cmd)) {

        case _IOC_NR(SVGALIB_HELPER_IOCSREPOUTB):
            copy_from_user(&iostr,(char *)arg,sizeof(iostr));
            if ( iostr.length > 768 ) return -EINVAL;
	    if ( (outb_str = kmalloc(iostr.length, GFP_KERNEL )) == NULL ) return -ENOMEM;
	    copy_from_user(outb_str,iostr.string,iostr.length);
	    check_io_range(iostr.port,minor);
	    for ( i = 0; i < iostr.length; i++ )
                outb ( outb_str[i], iostr.port );
            kfree ( outb_str );
            break;

        case _IOC_NR(SVGALIB_HELPER_IOCSOUTB):
           copy_from_user(&iov,(char *)arg,sizeof(iov));
           check_io_range(iov.port,minor);
           outb(iov.val,iov.port);
        break;

        case _IOC_NR(SVGALIB_HELPER_IOCSOUTW):
           copy_from_user(&iov,(char *)arg,sizeof(iov));
           check_io_range(iov.port,minor);
           outw(iov.val,iov.port);
        break;

        case _IOC_NR(SVGALIB_HELPER_IOCSOUTL):
           copy_from_user(&iov,(char *)arg,sizeof(iov));
           check_io_range(iov.port,minor);
           outl(iov.val,iov.port);
        break;

        case _IOC_NR(SVGALIB_HELPER_IOCGINB):
           copy_from_user(&iov,(char *)arg,sizeof(iov));
           check_io_range(iov.port,minor);
           iov.val=inb(iov.port);
           copy_to_user((char *)arg,&iov,sizeof(iov));
        break;

        case _IOC_NR(SVGALIB_HELPER_IOCGINW):
           copy_from_user(&iov,(char *)arg,sizeof(iov));
           check_io_range(iov.port,minor);
           iov.val=inw(iov.port);
           copy_to_user((char *)arg,&iov,sizeof(iov));
        break;

        case _IOC_NR(SVGALIB_HELPER_IOCGINL):
           copy_from_user(&iov,(char *)arg,sizeof(iov));
           check_io_range(iov.port,minor);
           iov.val=inl(iov.port);
           copy_to_user((char *)arg,&iov,sizeof(iov));
        break;

        case _IOC_NR(SVGALIB_HELPER_IOCGPCIINB):
            copy_from_user(&pciv,(char *)arg,sizeof(pciv));
            pdev = get_pci_dev(pciv.pcipos, minor);
            if(!pdev) return -EINVAL;
            pci_read_config_byte(pdev, pciv.address, &pb);
            pciv.val=pb;
            copy_to_user((char *)arg,&pciv,sizeof(pciv));
            break;

        case _IOC_NR(SVGALIB_HELPER_IOCGPCIINW):
            copy_from_user(&pciv,(char *)arg,sizeof(pciv));
            pdev = get_pci_dev(pciv.pcipos, minor);
            if(!pdev) return -EINVAL;
            pci_read_config_word(pdev, pciv.address, &pw);
            pciv.val=pw;
            copy_to_user((char *)arg,&pciv,sizeof(pciv));
            break;

        case _IOC_NR(SVGALIB_HELPER_IOCGPCIINL):
            copy_from_user(&pciv,(char *)arg,sizeof(pciv));
            pdev = get_pci_dev(pciv.pcipos, minor);
            if(!pdev) return -EINVAL;
            pci_read_config_dword(pdev, pciv.address, &pl);
            pciv.val=pl;
            copy_to_user((char *)arg,&pciv,sizeof(pciv));
            break;

        case _IOC_NR(SVGALIB_HELPER_IOCSPCIOUTB):
            copy_from_user(&pciv,(char *)arg,sizeof(pciv));
            pdev = get_pci_dev(pciv.pcipos, minor);
            if(!pdev) return -EINVAL;
            pb=pciv.val;
            pci_write_config_byte(pdev, pciv.address, pb);
            break;

        case _IOC_NR(SVGALIB_HELPER_IOCSPCIOUTW):
            copy_from_user(&pciv,(char *)arg,sizeof(pciv));
            pdev = get_pci_dev(pciv.pcipos, minor);
            if(!pdev) return -EINVAL;
            pw=pciv.val;
            pci_write_config_word(pdev, pciv.address, pw);
            break;

        case _IOC_NR(SVGALIB_HELPER_IOCSPCIOUTL):
            copy_from_user(&pciv,(char *)arg,sizeof(pciv));
            pdev = get_pci_dev(pciv.pcipos, minor);
            if(!pdev) return -EINVAL;
            pl=pciv.val;
            pci_write_config_dword(pdev, pciv.address, pl);
            break;

        default: 
            return -EINVAL;
    }
    return 0;
}   			


static int svgalib_helper_open( struct inode *inode, struct file * filp) {

   int minor = MINOR(inode->i_rdev);
   
   if(minor>=num_devices) return -ENODEV;
   
   MOD_INC_USE_COUNT;
   
   return 0;
}

static int svgalib_helper_release( struct inode *inode, struct file *filp) {
   MOD_DEC_USE_COUNT;
   
   return 0;
}

static int svgalib_helper_mmap(struct file *filp, struct vm_area_struct *vma) {
   int start=vma->vm_start;
   int end=vma->vm_end;
   int minor = MINOR(filp->f_dentry->d_inode->i_rdev);
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,3,0)
   int ofs=vma->vm_pgoff*PAGE_SIZE;
#else
   int ofs=vma->vm_offset;
#endif
   int i,j;

   if(minor==0) {
       if(((ofs>=0xa0000)&&(ofs+(end-start)<=0x100000))||
          ((ofs==0)&&(start==0)&&(end==0x1000))) {
#if defined(__powerpc__)
	  pgprot_val(vma->vm_page_prot) |= _PAGE_NO_CACHE|_PAGE_GUARDED;
#elif defined(__i386__)
	  if (boot_cpu_data.x86 > 3)
             pgprot_val(vma->vm_page_prot) |= _PAGE_PCD;
#endif
          if (remap_page_range(start, ofs, end-start, 
                               vma->vm_page_prot)) return -EAGAIN;
          if(ofs>=0xa0000)vma->vm_flags|=VM_IO;
          return 0;
       }
       for(i=1;i<num_devices;i++)
           for(j=0;j<6;j++) if(sh_pci_devs[i]->mem[j])
               if((ofs>=sh_pci_devs[i]->mem[j])&&
                   (ofs+(end-start)<=sh_pci_devs[i]->mem[j]+sh_pci_devs[i]->len[j])) {
#if defined(__powerpc__)
	  	   pgprot_val(vma->vm_page_prot) |= _PAGE_NO_CACHE|_PAGE_GUARDED;
#elif defined(__i386__)
                   if (boot_cpu_data.x86 > 3)
                      pgprot_val(vma->vm_page_prot) |= _PAGE_PCD;
#endif
                   if (remap_page_range(start, ofs, end-start, 
                                        vma->vm_page_prot)) return -EAGAIN;
                   return 0;
               }
   } else if(minor<num_devices) {
      	   i=minor;	
           for(j=0;j<6;j++) if(sh_pci_devs[i]->mem[j])
               if((ofs>=sh_pci_devs[i]->mem[j])&&
                   (ofs+(end-start)<=sh_pci_devs[i]->mem[j]+sh_pci_devs[i]->len[j])) {
#if defined(__powerpc__)
	  	   pgprot_val(vma->vm_page_prot) |= _PAGE_NO_CACHE|_PAGE_GUARDED;
#elif defined(__i386__)
                   if (boot_cpu_data.x86 > 3)
                      pgprot_val(vma->vm_page_prot) |= _PAGE_PCD;
#endif                   
                   if (remap_page_range(start, ofs, end-start, 
                                        vma->vm_page_prot)) return -EAGAIN;
                   return 0;
               }
   }
   return -EAGAIN;
}

struct file_operations svgalib_helper_fops = {
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,3,0)
   owner:   THIS_MODULE,
#endif
   ioctl:   svgalib_helper_ioctl,
   mmap:    svgalib_helper_mmap,	/* mmap */
   open:    svgalib_helper_open,	/* open */
   release: svgalib_helper_release	/* release */
};

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,3,0)
#define base_address(i) dev->resource[i].start
#else
#define base_address(i) (dev->base_address[i]&PCI_BASE_ADDRESS_MEM_MASK)
#endif

int init_module(void)
{
    int result, i;
    struct pci_dev *dev=NULL;
#ifdef CONFIG_DEVFS_FS
    devfs_handle_t slave_handle;
#endif
    /*
     * Register your major, and accept a dynamic number
     */
     
    printk(KERN_INFO "svgalib_helper: Initializing, version 1.9.14\n");
    result = devfs_register_chrdev(SVGALIB_HELPER_MAJOR, "svgalib_helper", &svgalib_helper_fops);
    if (result < 0) {
        printk(KERN_WARNING "svgalib_helper: can't get major %d\n",SVGALIB_HELPER_MAJOR);
        return result;
    }

    if((sh_pci_devs[0]=kmalloc(sizeof(struct sh_pci_device),GFP_KERNEL))==NULL) {
        goto nomem_error;
    }
    memset(sh_pci_devs[0],0,sizeof(struct sh_pci_device));
    num_devices=1;
    for(i=1;i<SVGALIB_HELPER_MAX_NR_DEVICES;i++) sh_pci_devs[i]=NULL;

#ifdef CONFIG_DEVFS_FS
    devfs_handle = devfs_mk_dir ( NULL, "svga_helper", NULL );
    devfs_register_series( devfs_handle,
               "%u", 8, DEVFS_FL_DEFAULT, SVGALIB_HELPER_MAJOR, 0,
                       S_IFCHR |  S_IRUGO | S_IRWXU, &svgalib_helper_fops, NULL ) ;
    devfs_mk_symlink( NULL, "svga", 0, "svga_helper/0", &slave_handle, NULL );
    devfs_auto_unregister( devfs_handle, slave_handle );
#endif /* devfsd support */

    if(pci_present()) {
        while((dev=pci_find_class(PCI_CLASS_DISPLAY_VGA<<8,dev)) && 
              (num_devices<=SVGALIB_HELPER_MAX_NR_DEVICES)) {
            if((sh_pci_devs[num_devices]=kmalloc(sizeof(struct sh_pci_device),GFP_KERNEL))==NULL) {
                goto nomem_error;
            }
            memset(sh_pci_devs[num_devices],0,sizeof(struct sh_pci_device));
            sh_pci_devs[num_devices]->dev=dev;
            pci_read_config_word(dev,0,&sh_pci_devs[num_devices]->vendor);
            pci_read_config_word(dev,2,&sh_pci_devs[num_devices]->id);
            pci_read_config_byte(dev,8,&sh_pci_devs[num_devices]->revision);
            printk(KERN_DEBUG "svgalib_helper: device%d: vendor:%.4x id:%.4x\n",num_devices,
            sh_pci_devs[num_devices]->vendor,sh_pci_devs[num_devices]->id);
            for(i=0;i<6;i++){
                unsigned int t; 
                int len;
                pci_read_config_dword(dev,16+4*i,&result);
                if(result) {
                    pci_write_config_dword(dev,16+4*i,0xffffffff);
                    pci_read_config_dword(dev,16+4*i,&t);
                    pci_write_config_dword(dev,16+4*i,result);
                    len = ~(t&~0xf)+1;
                    if (len){
                       sh_pci_devs[num_devices]->mem[i]=result&~0xf;
                       sh_pci_devs[num_devices]->flags[i]=0x80 | (result&0xf);
                       sh_pci_devs[num_devices]->len[i]=len;
                       sh_pci_devs[num_devices]->mask[i]=t&~0xf;
                       printk(KERN_DEBUG "device%d: region%d, base=%.8x len=%d type=%d\n",
                       num_devices,i, result&(~0xf), len, result&0xf);
                    }
                }
            }
            num_devices++;
        }
    }

    EXPORT_NO_SYMBOLS;

    return 0; /* succeed */

    nomem_error:
        for(i=0;i<SVGALIB_HELPER_MAX_NR_DEVICES;i++) 
            if(sh_pci_devs[i])kfree(sh_pci_devs[i]);     
    devfs_unregister_chrdev(SVGALIB_HELPER_MAJOR, "svgalib_helper");
    return result;
}

void cleanup_module(void)
{
    int i;
    for(i=0;i<SVGALIB_HELPER_MAX_NR_DEVICES;i++) 
        if(sh_pci_devs[i]) {
            kfree(sh_pci_devs[i]);
        }
     
#ifdef CONFIG_DEVFS_FS
    devfs_unregister(devfs_handle);
#endif
    devfs_unregister_chrdev(SVGALIB_HELPER_MAJOR, "svgalib_helper");
}

#ifdef MODULE_LICENSE
MODULE_LICENSE("GPL and additional rights");
#endif

