#define MMIO_IN8(addr)  (*(MMIO_POINTER + addr))
#define MMIO_IN32(addr)  LE32(*(unsigned int *)(MMIO_POINTER + addr))
#define MMIO_OUT8(addr, val)  (*(unsigned char *)(MMIO_POINTER + addr) = val)
#define MMIO_OUT32(addr, val)  (*(unsigned int *)(MMIO_POINTER + addr) = LE32(val))
#define INREG(addr)         MMIO_IN32(addr)
#define INREG8(addr)         MMIO_IN8(addr)
#define OUTREG8(addr, val)  MMIO_OUT8(addr, val)
#define OUTREG(addr, val)   MMIO_OUT32(addr, val)
#define LE32(x) (x)

#define R128_CLOCK_CNTL_DATA              0x000c
#define R128_CLOCK_CNTL_INDEX             0x0008
#       define R128_PLL_WR_EN             (1 << 7)
#       define R128_PLL_DIV_SEL           (3 << 8)
#define R128_PPLL_REF_DIV                 0x0003 /* PLL */
#       define R128_PPLL_REF_DIV_MASK     0x03ff
#       define R128_PPLL_ATOMIC_UPDATE_R  (1 << 15) /* same as _W */
#       define R128_PPLL_ATOMIC_UPDATE_W  (1 << 15) /* same as _R */
#define R128_PPLL_CNTL                    0x0002 /* PLL */
#       define R128_PPLL_RESET                (1 <<  0)
#       define R128_PPLL_SLEEP                (1 <<  1)
#       define R128_PPLL_ATOMIC_UPDATE_EN     (1 << 16)
#       define R128_PPLL_VGA_ATOMIC_UPDATE_EN (1 << 17)
#define R128_PPLL_DIV_1                   0x0005 /* PLL */
#define R128_PPLL_DIV_3                   0x0007 /* PLL */
#       define R128_PPLL_FB3_DIV_MASK     0x07ff
#       define R128_PPLL_POST3_DIV_MASK   0x00070000
#define R128_HTOTAL_CNTL                  0x0009 /* PLL */

static int R128Div(int n, int d)
{
    return (n + (d / 2)) / d;
}

#define OUTREGP(addr, val, mask)   \
    do {                           \
	uint32_t tmp = INREG(addr);\
	tmp &= (mask);             \
	tmp |= (val);              \
	OUTREG(addr, tmp);         \
    } while (0)

#define OUTPLL(addr, val)                                                 \
    do {                                                                  \
	OUTREG8(R128_CLOCK_CNTL_INDEX, ((addr) & 0x1f) | R128_PLL_WR_EN); \
	OUTREG(R128_CLOCK_CNTL_DATA, val);                                \
    } while (0)

static unsigned INPLL(int addr)
{
    OUTREG8(R128_CLOCK_CNTL_INDEX, addr & 0x1f);
    return INREG(R128_CLOCK_CNTL_DATA);
}

typedef struct {
    uint16_t        reference_freq;
    uint16_t        reference_div;
    uint32_t        min_pll_freq;
    uint32_t        max_pll_freq;
} R128PLLRec, *R128PLLPtr;

static void R128InitPLLRegisters(R128PLLPtr pll,
				 double dot_clock, int *div3)
{
    unsigned long freq = dot_clock / 10;
    struct {
	int divider;
	int bitvalue;
    } *post_div,
      post_divs[]   = {
	{  1, 0 },              /* VCLK_SRC                 */
	{  2, 1 },              /* VCLK_SRC/2               */
	{  4, 2 },              /* VCLK_SRC/4               */
	{  8, 3 },              /* VCLK_SRC/8               */
	{  3, 4 },              /* VCLK_SRC/3               */
	{  6, 6 },              /* VCLK_SRC/6               */
	{ 12, 7 },              /* VCLK_SRC/12              */
	{  0, 0 }
    };
    int pll_output_freq, feedback_div;

    if (freq > pll->max_pll_freq)      freq = pll->max_pll_freq;
    if (freq * 12 < pll->min_pll_freq) freq = pll->min_pll_freq / 12;

    for (post_div = &post_divs[0]; post_div->divider; ++post_div) {
	pll_output_freq = post_div->divider * freq;
	if (pll_output_freq >= pll->min_pll_freq
	    && pll_output_freq <= pll->max_pll_freq) break;
    }

    feedback_div   = R128Div(pll->reference_div * pll_output_freq,
				   pll->reference_freq);
    *div3     = feedback_div | (post_div->bitvalue << 16);
}

int r128_setclock(int clock) {
    R128PLLRec pll;
    int div3, i;

    pll.reference_freq = 2700;
    pll.min_pll_freq   = 20000;
    pll.max_pll_freq   = 35000;
    pll.reference_div =	INPLL(R128_PPLL_REF_DIV) & R128_PPLL_REF_DIV_MASK;
   
    R128InitPLLRegisters(&pll, clock, &div3);

printf("%i   %x\n",pll.reference_div, div3);
 
    OUTPLL(6, div3);
    usleep(1);
    
    i=inb(0x3cc);
    i&=0xf3;
    outb(i|0x08, 0x3c2);
}

