#define LDT_DEBUG 1
#define DEVNAME "BCM-LDT"

#include <linux/config.h>
#include <linux/module.h>
#include <linux/version.h>

#include <linux/init.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/slab.h>
#include <linux/delay.h>
#include <linux/interrupt.h>

#include <asm/uaccess.h>
#include <asm/io.h>
#include <asm/sibyte/swarm.h>
#include <asm/sibyte/sb1250.h>
#include <asm/sibyte/sb1250_regs.h>
#include <asm/sibyte/sb1250_uart.h>
#include <asm/sibyte/sb1250_int.h>
#include <asm/sibyte/sb1250_dma.h>
#include <asm/sibyte/sb1250_ldt.h>
#include <asm/sibyte/64bit.h>

#include <asm/system.h>   /* cli(), *_flags */
#include <asm/segment.h>  /* memcpy and such */
#include <asm/pgtable.h>

#include "ldtreg.h"
#include "pcireg.h"

/* PCI regions in LOCAL (ZBbus) space, match bytes policy.  See Figure 37. */

#define PCI_IO_SPACE		A_PHYS_LDTPCI_IO_MATCH_BYTES    /* 0x00dc000000 */
#define	PCI_IO_SPACE_SIZE	0x0002000000

#define PCI_MEM_SPACE		A_PHYS_LDTPCI_IO_MATCH_BYTES_32 /* 0x0040000000 */
#define PCI_MEM_SPACE_SIZE	0x0020000000

#define PCI_CONF_SPACE		A_PHYS_LDTPCI_CFG_MATCH_BYTES   /* 0x00de000000 */
#define PCI_CONF_SPACE_SIZE	0x0001000000

#define	SB1250_BIT_ENDIAN_OFFSET	0x0020000000

/*
 * NB: initial limits for address allocation are assumed to be aligned
 * appropriately for PCI bridges (4K boundaries for I/O, 1M for memory).
 */

static volatile u32 * pciconfig=0;
static volatile u32 * ldtregs=0;
static volatile u16 * pciconfig16;
static volatile u8 * pciconfig8;
static volatile u32 * EOI;
static volatile u8 * ldtflash;
static volatile u32 * ldtconf;

static int sb1250_ldt_slave_mode;
static int sb1250_ldt_init;   /* Set to one after LHB sees InitDone */

static void *bdev=(void *)0x1234fedc;

static int verbose=1;
static int linkfreq=400;
static int master=0;
MODULE_PARM(master, "i");
MODULE_PARM(verbose, "i");
MODULE_PARM(linkfreq, "i");
/* The pass 1 BCM1250 does not implement EOI cycles correctly.  Later
   passes do.  The following variable controls whether PCI interrupt
   mappings are programmed to be level-sensitive (EOI) or
   edge-sensitive (no EOI) in LDT-PCI bridges. */

static int eoi_implemented=1;


/* exported variables */
int ldt_master;
void * ldt_buf;

#define	SB1250_PCI_MAKE_TAG(b,d,f)					\
    (((b) << 16) | ((d) << 11) | ((f) << 8))
#define	SB1250_CFG_BASE							\
    PHYS_TO_XKSEG_UNCACHED(PCI_CONF_SPACE + SB1250_BIT_ENDIAN_OFFSET)

#if defined(__MIPSEB)
/* This is for big-endian with a match bits policy. */
#define	SB1250_CFG_ADDR(t, o, w)					\
    ((SB1250_CFG_BASE + (t) + (o)) ^ (4 - (w)))
#elif defined(__MIPSEL)
/* This is for little-endian, either policy. */
#define	SB1250_CFG_ADDR(t, o, w)					\
    (SB1250_CFG_BASE + (t) + (o))
#else
#error "Must specifiy either MIPSEL or MIPSEB"
#endif


/* The SB-1250 integrated host bridges. */

#define PCI_VENDOR_SIBYTE               0x166d

#define	SB1250_PCI_BRIDGE	(SB1250_PCI_MAKE_TAG(0,0,0))
#define	SB1250_LDT_BRIDGE	(SB1250_PCI_MAKE_TAG(0,1,0))

#define PCI_FLG_NORMAL        0x00000001
#define PCI_FLG_VERBOSE       0x00000003
#define PCI_FLG_LDT_PREFETCH  0x00000004
	
/* Implementation-specific registers for the LDT Host Bridge (LHB) */

#define LHB_LINK_BASE                   0x40

#define LHB_LINKCMD_REG                 (LHB_LINK_BASE+LDT_COMMAND_CAP_OFF)

#define LHB_LINKCTRL_REG                (LHB_LINK_BASE+LDT_LINK_OFF(0))

#define LHB_LINKCTRL_CRCERROR           (1 << LDT_LINKCTRL_CRCERROR_SHIFT)
#define LHB_LINKCTRL_ERRORS             (LHB_LINKCTRL_CRCERROR | \
                                         LDT_LINKCTRL_LINKFAIL)

#define LHB_LINKFREQ_REG                (LHB_LINK_BASE+LDT_FREQ_OFF)
#define LHB_LFREQ_REG                   (LHB_LINKFREQ_REG+1)

#define LHB_LFREQ_200                   LDT_FREQ_200        /* 200 MHz */
#define LHB_LFREQ_300                   LDT_FREQ_300        /* 300 MHz */
#define LHB_LFREQ_400                   LDT_FREQ_400        /* 400 MHz */
#define LHB_LFREQ_500                   LDT_FREQ_500        /* 500 MHz */
#define LHB_LFREQ_600                   LDT_FREQ_600        /* 600 MHz */

#define LHB_SRI_CMD_REG                 0x50

#define LHB_SRI_CMD_SIPREADY            (1 << 16)
#define LHB_SRI_CMD_SYNCPTRCTL          (1 << 17)
#define LHB_SRI_CMD_REDUCESYNCZERO      (1 << 18)
#define LHB_SRI_CMD_DISSTARVATIONCNT    (1 << 19)
#define LHB_SRI_CMD_RXMARGIN_SHIFT      20
#define LHB_SRI_CMD_RXMARGIN_MASK       (0x1f << LHB_SRI_CMD_RXMARGIN_SHIFT)
#define LHB_SRI_CMD_PLLCOMPAT           (1 << 25)
#define LHB_SRI_CMD_TXOFFSET_SHIFT      28
#define LHB_SRI_CMD_TXOFFSET_MASK       (0x7 << LHB_SRI_CMD_TXOFFSET_SHIFT)
#define LHB_SRI_CMD_LINKFREQDIRECT      (1 << 31)

#define LHB_SRI_TXNUM_REG               0x54
#define LHB_SRI_RXNUM_REG               0x58

#define LHB_ERR_CTRL_REG                0x68

#define LHB_ERR_CTRL                    0x00ffffff
#define LHB_ERR_STATUS                  0xff000000

#define LHB_SRI_CTRL_REG                0x6c

#define LHB_ASTAT_REG                   0x70

#define LHB_ASTAT_TGTDONE_SHIFT         0
#define LHB_ASTAT_TGTDONE_MASK          (0xFF << LHB_ASTAT_TGTDONE_SHIFT)

#define LHB_TXBUFCNT_REG                0xc8



#define pci_tagprintf(tag, regs...) printk(KERN_DEBUG regs)
typedef u32 pcireg_t;
typedef int pcitag_t;
typedef u64 physaddr_t;

static pcireg_t
pci_conf_readn(pcitag_t tag, int reg, int width)
{
    physaddr_t addrp;
    pcireg_t data;

    addrp = tag+reg;
    switch (width) {
    case 1:
        data = (pcireg_t) pciconfig8[addrp];
        break;
    case 2:
        data = (pcireg_t) pciconfig16[addrp>>1];
        break;
    default:
    case 4:
        data = (pcireg_t) pciconfig[addrp>>2];
        break;
    }

    return data;
}

pcireg_t
pci_conf_read8(pcitag_t tag, int reg)
{
    return pci_conf_readn(tag, reg, 1);
}

pcireg_t
pci_conf_read16(pcitag_t tag, int reg)
{
    return pci_conf_readn(tag, reg, 2);
}

#ifndef pci_conf_read32
pcireg_t
pci_conf_read32(pcitag_t tag, int reg)
{
    return pci_conf_readn(tag, reg, 4);
}
#endif

pcireg_t
pci_conf_read(pcitag_t tag, int reg)
{
    return pci_conf_readn(tag, reg, 4);
}

static void
pci_conf_writen(pcitag_t tag, int reg, pcireg_t data, int width)
{
    physaddr_t addrp;
//    mips_wbflush();

    addrp = tag+reg;
    switch (width) {
    case 1:
        pciconfig8[addrp]=data;
        break;
    case 2:
        pciconfig16[addrp>>1]=data;
        break;
    default:
    case 4:
        pciconfig[addrp>>2]=data;
        break;
    }

//    mips_wbflush();
}

void
pci_conf_write8(pcitag_t tag, int reg, pcireg_t data)
{
    pci_conf_writen(tag, reg, data, 1);
}

void
pci_conf_write16(pcitag_t tag, int reg, pcireg_t data)
{
    pci_conf_writen(tag, reg, data, 2);
}

#ifndef pci_conf_write32
void
pci_conf_write32(pcitag_t tag, int reg, pcireg_t data)
{
    pci_conf_writen(tag, reg, data, 4);
}
#endif

void
pci_conf_write(pcitag_t tag, int reg, pcireg_t data)
{
    pci_conf_writen(tag, reg, data, 4);
}



#if (LDT_DEBUG > 1)
static void
show_ldt_status (void)
{
    pcireg_t cmd;

    cmd = pci_conf_read32(SB1250_LDT_BRIDGE, LHB_SRI_CMD_REG);
    printk(" SriCmd %04x\n", (cmd >> 16) & 0xffff);
    printk(" TXNum %08x TxDen %02x  RxNum %08x RxDen %02x ErrCtl %08x\n",
	    pci_conf_read32(SB1250_LDT_BRIDGE, LHB_SRI_TXNUM_REG),
	    cmd & 0xff,
	    pci_conf_read32(SB1250_LDT_BRIDGE, LHB_SRI_RXNUM_REG),
	    (cmd >> 8) & 0xff,
	    pci_conf_read32(SB1250_LDT_BRIDGE, LHB_ERR_CTRL_REG));
    printk(" LDTCmd %08x LDTCfg %08x LDTFreq %08x\n",
	    pci_conf_read32(SB1250_LDT_BRIDGE, LHB_LINKCMD_REG),
	    pci_conf_read32(SB1250_LDT_BRIDGE, LHB_LINKCTRL_REG),
	    pci_conf_read32(SB1250_LDT_BRIDGE, LHB_LINKFREQ_REG));
}
#else
#define show_ldt_status() ((void)0)
#endif /* LDT_DEBUG */

/*
 * Assert warm reset for the given number of ticks.
 */
static void
lhb_link_reset (int delay)
{
    pcireg_t prev, cmd;
    pcireg_t brctrl;

    prev = pci_conf_read32(SB1250_LDT_BRIDGE, LHB_LINKCMD_REG);
    cmd = prev | LDT_COMMAND_WARM_RESET;
    pci_conf_write32(SB1250_LDT_BRIDGE, LHB_LINKCMD_REG, cmd);

    brctrl = pci_conf_read32(SB1250_LDT_BRIDGE, PPB_BRCTL_INTERRUPT_REG);
    brctrl |= PPB_BRCTL_SECONDARY_RESET;
    pci_conf_write32(SB1250_LDT_BRIDGE, PPB_BRCTL_INTERRUPT_REG, brctrl);

    mdelay(100);

    brctrl &=~ PPB_BRCTL_SECONDARY_RESET;
    pci_conf_write32(SB1250_LDT_BRIDGE, PPB_BRCTL_INTERRUPT_REG, brctrl);
    brctrl = pci_conf_read32(SB1250_LDT_BRIDGE, PPB_BRCTL_INTERRUPT_REG);

    pci_conf_write32(SB1250_LDT_BRIDGE, LHB_LINKCMD_REG, prev);
}

/*
 * Poll for InitDone on LHB's outgoing link.
 */
static int
lhb_link_ready (int maxpoll)
{
    volatile pcireg_t ctrl;
    int count;
    int linkerr;

    count = 0;
    linkerr = 0;
    ctrl = 0;

    while ((ctrl & (LDT_LINKCTRL_INITDONE | LDT_LINKCTRL_LINKFAIL)) == 0
	   && count < maxpoll) {
        ctrl = pci_conf_read32(SB1250_LDT_BRIDGE, LHB_LINKCTRL_REG);
	count++;
	if ((ctrl & LHB_LINKCTRL_ERRORS) != 0 && !linkerr) {
	    if (verbose > PCI_FLG_NORMAL)
		pci_tagprintf(SB1250_LDT_BRIDGE,
			      "LDT Err, count %d Err = 0x%04x\n",
			      count, ctrl & 0xffff);
	    linkerr = 1;
	}
	if (count == maxpoll) {
	    if (verbose > PCI_FLG_NORMAL)
		pci_tagprintf(SB1250_LDT_BRIDGE, "Link timeout\n");
	    linkerr = 1;
	}

    }

    if (verbose > PCI_FLG_NORMAL)
	pci_tagprintf(SB1250_LDT_BRIDGE, "lhb_link_ready: count %d\n", count);
    return linkerr;
}

static void
lhb_null_config (void)
{
    /* Even if the LDT fabric is not to be initialized by us, we must
       write the bus number, base and limit registers in the host
       bridge for proper operation (see 8.11.1) */
    pcireg_t iodata;
    pcireg_t memdata;

    /* The primary bus is 0.  Set secondary, subordinate to 0 also. */
    pci_conf_write32(SB1250_LDT_BRIDGE, PPB_BUSINFO_REG, 0);

    iodata = pci_conf_read32(SB1250_LDT_BRIDGE, PPB_IO_STATUS_REG);
    iodata &=~ (PPB_IO_BASE_MASK | PPB_IO_LIMIT_MASK);
    iodata |= (1 << 4) | (0 << (8+4));
    pci_conf_write32(SB1250_LDT_BRIDGE, PPB_IO_STATUS_REG, iodata);
    iodata = 0x0000f200;   /* recommended value */
    pci_conf_write32(SB1250_LDT_BRIDGE, PPB_IO_UPPER_REG, iodata);

    memdata = pci_conf_read32(SB1250_LDT_BRIDGE, PPB_MEM_REG);
    memdata &=~ (PPB_MEM_BASE_MASK | PPB_MEM_LIMIT_MASK);
    memdata |= (1 << 4) | (0 << (16+4));  /* empty window */
    pci_conf_write32(SB1250_LDT_BRIDGE, PPB_MEM_REG, memdata);
}


/*
 * LDT host bridge initialization.
 */
static void
lhb_init (unsigned linkfreq)
{
    int i;
    pcireg_t cr;
    pcireg_t sri_cmd;
    volatile pcireg_t t;      /* used for reads that push writes */
    uint8_t ldt_freq;
    int linkerr;
    int retry;

    sb1250_ldt_init = 0;

    /* Clear any pending error bits */
    t = pci_conf_read32(SB1250_LDT_BRIDGE, LHB_ERR_CTRL_REG);
    pci_conf_write32(SB1250_LDT_BRIDGE, LHB_ERR_CTRL_REG, t | LHB_ERR_STATUS);

    cr = pci_conf_read32(SB1250_LDT_BRIDGE, PCI_CLASS_REG);
    eoi_implemented = (PCI_REVISION(cr) >= 2);

    /* First set up System Reset Initialization registers (Table
       8-12).  This code is designed to be run following a full reset.
       After a chip warm reset (SipReady already set), most of it is
       skipped.  Depending on previous history, that can leave things
       in an inconsistent state, but there's no recovery from that
       short of a chip cold reset. */

    t = pci_conf_read32(SB1250_LDT_BRIDGE, LHB_SRI_CMD_REG);
    if (t & LHB_SRI_CMD_SIPREADY) {
        pci_tagprintf(SB1250_LDT_BRIDGE, "Warning: SipReady already set\n");
	/* Try just doing a warm reset. */
		mdelay(100);
		goto finish;
    }

    /* LDT 0.17 compatibility mode:
         SriCmd = (!LinkFreqDirect, LdtPLLCompat, [DisStarveCnt,]
                   TxInitialOffset=5,
                   RxMargin=2,
		   !SipReady)
         SriRxDen = 0x10
         SriTxDen = 0x10
    */
    sri_cmd = LHB_SRI_CMD_PLLCOMPAT;

    /* Empirically, RxMargin is not critical with a 200 MHz LDT clock
       but must be small (less than 15, and 0 seems reliable) with a
       400 MHz LDT clock.  Current default is 2. */
    sri_cmd |= ((2 << LHB_SRI_CMD_RXMARGIN_SHIFT) |
		(5 << LHB_SRI_CMD_TXOFFSET_SHIFT) |
		0x1010);        /* Rx/TxDen defaults */

    /* Setting DisStarveCnt is recommended for Pass 1 parts. */
    if (PCI_REVISION(cr) == 1)
	sri_cmd |= LHB_SRI_CMD_DISSTARVATIONCNT;

    /* Map the link frequency to a supported value.  Note: we assume
       that the CPU and IOB0 clocks are set high enough for the
       selected frequency.  In LDT 0.17 compatibility mode, this is
       the final frequency.  */
    if (linkfreq < 200)
	linkfreq = 200;
    else if (linkfreq > 600)
	linkfreq = 600;
#if 0
    linkfreq = ((linkfreq + (25-1))/50) * 50;
#else
    /* Passes 1 and 2 do not support LINKFREQDIRECT. Force a standard value. */
    linkfreq = ((linkfreq + (50-1))/100) * 100;
#endif
    printk(KERN_INFO "HyperTransport: %d MHz\n", linkfreq);

    if (linkfreq % 100 == 0) {
	/* Encode supported standard values per the LDT spec. */      
	switch (linkfreq) {
	default:
	case 200:  ldt_freq = LHB_LFREQ_200;  break;
	case 300:  ldt_freq = LHB_LFREQ_300;  break;
	case 400:  ldt_freq = LHB_LFREQ_400;  break;
	case 500:  ldt_freq = LHB_LFREQ_500;  break;
	case 600:  ldt_freq = LHB_LFREQ_600;  break;
	}
    } else {
	/* Compute PLL ratio for 100 MHz reference in 3b1 format (Table 23) */
	sri_cmd |= LHB_SRI_CMD_LINKFREQDIRECT;
	ldt_freq = linkfreq / 50;
    }

    pci_conf_write32(SB1250_LDT_BRIDGE, LHB_SRI_CMD_REG, sri_cmd);

    /* Set the SRI dividers */
    pci_conf_write32(SB1250_LDT_BRIDGE, LHB_SRI_TXNUM_REG, 0x0000ffff);
    pci_conf_write32(SB1250_LDT_BRIDGE, LHB_SRI_RXNUM_REG, 0x0000ffff);
    t = pci_conf_read32(SB1250_LDT_BRIDGE, LHB_SRI_RXNUM_REG); /* push */

    /* Directed test: SPIN(10) here */
    for (i = 0; i < 10; i++)
      t = pci_conf_read32(SB1250_LDT_BRIDGE, PCI_ID_REG);

    if (linkfreq == 200) {
	/* This code demonstrates a Pass 1 workaround.  Leaving the bridge
	   link frequency at the default 200 MHz is not reliable.  If 200
	   MHz is desired, set to 400 to poke the PLL, then set back to
	   200 */
	pci_conf_write8(SB1250_LDT_BRIDGE, LHB_LFREQ_REG, LHB_LFREQ_400);
	t = pci_conf_read8(SB1250_LDT_BRIDGE, LHB_LFREQ_REG);
	mdelay(500);
    }
    pci_conf_write8(SB1250_LDT_BRIDGE, LHB_LFREQ_REG, ldt_freq);

    /* Set the Error Control register (some fatal interrupts). */
    pci_conf_write32(SB1250_LDT_BRIDGE, LHB_ERR_CTRL_REG, 0x00001209);

    /* Set the SRI Xmit Control register (4 packets). */
#if 0
    pci_conf_write32(SB1250_LDT_BRIDGE, LHB_SRI_CTRL_REG, 0x00042525);
#else
    {
      const char * str;
      unsigned int nbufs;

      nbufs=16;
	  pci_conf_write32(SB1250_LDT_BRIDGE,
		       LHB_SRI_CTRL_REG, 0x00040000 | nbufs);
    }
#endif
   
    /* Set the Tx buffer size (16 buffers each). */
    pci_conf_write32(SB1250_LDT_BRIDGE, LHB_TXBUFCNT_REG, 0x00ffffff);

    /* Push the writes */
    t = pci_conf_read32(SB1250_LDT_BRIDGE, LHB_TXBUFCNT_REG); /* push */

    /* Indicate SIP Ready */
    sri_cmd |= LHB_SRI_CMD_SIPREADY;
    pci_conf_write32(SB1250_LDT_BRIDGE, LHB_SRI_CMD_REG, sri_cmd);
    t = pci_conf_read32(SB1250_LDT_BRIDGE, LHB_SRI_CMD_REG);  /* push */

    retry = (sb1250_ldt_slave_mode ? 4 : 1);
    for (;;) {

	/* wait for LinkFail or InitDone */
	linkerr = lhb_link_ready(1<<25);   /* empirical delay */

	t = pci_conf_read32(SB1250_LDT_BRIDGE, LHB_ERR_CTRL_REG);
	if ((t & LHB_ERR_STATUS) != 0) {
	    linkerr = 1;
	    if (verbose > PCI_FLG_NORMAL)
		pci_tagprintf(SB1250_LDT_BRIDGE,
			      "ErrStat = 0x%02x\n", t >> 24);
	    pci_conf_write32(SB1250_LDT_BRIDGE, LHB_ERR_CTRL_REG,
			     t | LHB_ERR_STATUS);
	}

	t = pci_conf_read32(SB1250_LDT_BRIDGE, LHB_LINKCTRL_REG);
	if ((t & LHB_LINKCTRL_ERRORS) != 0) {
	    linkerr = 1;
	    if (verbose > PCI_FLG_NORMAL)
		pci_tagprintf(SB1250_LDT_BRIDGE,
			      "LinkCtrl CRCErr = 0x%01x\n", (t >> 8) & 0xf);
	}

	if (!linkerr || retry == 0)
	    break;

	/* Clear errors in preparation for another try.  Delay long
           enough (below) for CRC errors to reappear; otherwise, the
           link can look good, but subsequent non-posted transactions
           will hang. */
	t = pci_conf_read32(SB1250_LDT_BRIDGE, LHB_LINKCTRL_REG);
	t |= LDT_LINKCTRL_CRCERROR_MASK;    /* Clear CrcErr bits */
	pci_conf_write32(SB1250_LDT_BRIDGE, LHB_LINKCTRL_REG, t);

	/* Try again.  Do a reset iff an LDT master, since a poorly
           timed reset by a slave will break any link initialization
           in progress. */
	retry--;

	if (sb1250_ldt_slave_mode)
	    mdelay(100);
	else
	    lhb_link_reset(HZ/10);
    }

 finish:
    t = pci_conf_read32(SB1250_LDT_BRIDGE, LHB_LINKCTRL_REG);

    if ((t & LDT_LINKCTRL_INITDONE) == 0) {
	printk(KERN_INFO "HyperTransport not initialized: InitDone not set\n");
	if (verbose > PCI_FLG_NORMAL)
	    pci_tagprintf(SB1250_LDT_BRIDGE,
			  "  Link Cmd = 0x%08x, Link Ctrl = 0x%08x\n",
			  pci_conf_read32(SB1250_LDT_BRIDGE, LHB_LINKCMD_REG),
			  pci_conf_read32(SB1250_LDT_BRIDGE, LHB_LINKCTRL_REG));
    } else if ((t & LHB_LINKCTRL_ERRORS) != 0) {
	printk(KERN_INFO "HyperTransport not initialized: "
		"LinkFail or CRCErr set, LinkCtrl = 0x%08x\n", t);
    } else if (!sb1250_ldt_slave_mode)
	sb1250_ldt_init = 1;

    /* Clear any pending error bits */
    t = pci_conf_read32(SB1250_LDT_BRIDGE, LHB_ERR_CTRL_REG);
    pci_conf_write32(SB1250_LDT_BRIDGE, LHB_ERR_CTRL_REG, t & 0xff000000);

    if (sb1250_ldt_slave_mode) {
	/* This is LDT slave mode.  The documentation is not very clear
	   on how much low level initialization should be done before
	   sleeping.  We just set Master Enable so that we can subsequently
           access LDT space. */
	pcireg_t cmd;

	/* If there are intermediate devices on the LDT, we would like
           our addressing to match the master's, but we don't know it
           and can't force it here.  Instead, we close all the windows
           into configurable space, which is at least safe. */
	lhb_null_config();

	cmd = pci_conf_read32(SB1250_LDT_BRIDGE, PCI_COMMAND_STATUS_REG);
	cmd &= (PCI_COMMAND_MASK << PCI_COMMAND_SHIFT);  /* preserve status */
	cmd |= PCI_COMMAND_MASTER_ENABLE;
	pci_conf_write32(SB1250_LDT_BRIDGE, PCI_COMMAND_STATUS_REG, cmd);
    } else if (!sb1250_ldt_init) {
	pcireg_t lr;
  
	lhb_null_config();

	/* Also, terminate the link */
	lr = pci_conf_read32(SB1250_LDT_BRIDGE, LHB_LINKCTRL_REG);
	lr |= LDT_LINKCTRL_EOC;
	pci_conf_write32(SB1250_LDT_BRIDGE, LHB_LINKCTRL_REG, lr);
	/* XXX API spec hints at needing a delay here. */
	lr |= LDT_LINKCTRL_TXOFF;
	pci_conf_write32(SB1250_LDT_BRIDGE, LHB_LINKCTRL_REG, lr);
    }

    show_ldt_status();
}


/*
 * Called to initialise IOB0 and the host bridges at the beginning of time.
 */
int
pci_hwinit ()
{
    if (master)
        sb1250_ldt_slave_mode = 0;
    else
        sb1250_ldt_slave_mode = 1;

    eoi_implemented = 0;   /* conservative default */

    /* stop the SB-1250 from servicing any further LDT requests */
    pci_conf_write32(SB1250_LDT_BRIDGE, PCI_COMMAND_STATUS_REG, 0);

    /* initialize the LDT host bridge */
    lhb_init(linkfreq);

    return 1;
}

#define SCRATCH_REG (IO_SPACE_BASE + 0x10020c10)

#define MBOX3_MASK 0xffff
#define MBOX (IO_SPACE_BASE + 0x100200c0)
#define MBOX_CLR (IO_SPACE_BASE + 0x100200d0)

void ldt_interrupt(int value) {
	/* value should be (0x40|num)<<13, where 0<=num<16 */
	EOI[(value&0x1fe000)<<1 | (value&3)]=value;
}

void (*ldt_callback)(int)=0;

static void mbox_handler(int irq, void *dev_id, struct pt_regs *regs) {
	int mbox3;
	
	printk(KERN_DEBUG "MBOX3 irq %i\n",irq);

	while ((mbox3=in64(MBOX)&MBOX3_MASK)) {
		if(mbox3 & 1) {
			/* startup interrupt */

			/* use */
			out64(MBOX_CLR, 1);
			EOI[(0<<14) | 3];

		}
		
		if(mbox3 & 2) {
			if(ldt_callback)ldt_callback(2);
			out64(MBOX_CLR, 2);
			EOI[(1<<14) | 3];
		}
		
		if(mbox3 & 4) {
			if(ldt_callback)ldt_callback(4);
			out64(MBOX_CLR, 4);
			EOI[(2<<14) | 3];
		}
		
		if(mbox3 & 8) {
			if(ldt_callback)ldt_callback(8);
			out64(MBOX_CLR, 8);
			EOI[(3<<14) | 3];
		}
	}
	
}


static int __init ldt_init(void) {
	int i;
	void *buf;
	u64 pbuf;
	u64 bufa;

	pciconfig=ioremap(0xfe000000ULL, 0x20000);
	pciconfig8=(u8 *)pciconfig;
	pciconfig16=(u16 *)pciconfig;
	ldtregs=ioremap(0x10020000ULL, 0x10000);
	EOI=ioremap(0xd8000000ULL, 0x1000000);
	ldtconf=ioremap( 0xe010020000, 0x10000);
	
	if(!master) {
		buf=kmalloc(0x10000, GFP_KERNEL);
		pbuf=virt_to_phys(buf);
		ldtregs[0xc10<<4]=pbuf&0xffffffff;
		ldtregs[0xc14<<4]=(pbuf>>32);
		printk(KERN_DEBUG "buf=%08x pbuf=%08x\n");
	}
	
	ldt_master=master;
	
	pci_hwinit();

	request_irq(K_INT_MBOX_3, mbox_handler, SA_SHIRQ, DEVNAME, bdev); /* dedicate MBOX3 to LDT */
	
	if(master) {
		/* interrupt slave - bit 0 */
		mdelay(100);
		ldt_interrupt(0);

		bufa=(u64)ldtconf[0x0c10>>2] + ((u64)ldtconf[0x0c14>>2]<<32);
		if(verbose)printk(KERN_DEBUG "bufa = %08Lx %08x\n", bufa, *ldtconf);
		ldt_buf = ioremap(bufa, 0x10000);
	} else {
		ldt_buf = buf;
	}
	
	return 0;
}

static void ldt_exit(void) {
	iounmap((void *)pciconfig);
	iounmap((void *)ldtregs);
	iounmap((void *)EOI);
	iounmap((void *)ldtconf);

	if(master) {
		iounmap(ldt_buf);
	} else {
		kfree(ldt_buf);
	}
	free_irq(K_INT_MBOX_3, bdev);
}

module_exit(ldt_exit);
module_init(ldt_init);

EXPORT_SYMBOL(ldt_callback);
EXPORT_SYMBOL(ldt_interrupt);
EXPORT_SYMBOL(ldt_master);
EXPORT_SYMBOL(ldt_buf);

