/*
 * $Id: flash.c,v 1.6 2002/11/05 11:44:58 telka Exp $
 *
 * Copyright (C) 2002 ETC s.r.o.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 *
 * Written by Marcel Telka <marcel@telka.sk>, 2002.
 *
 * Documentation:
 * [1] Advanced Micro Devices, "Common Flash Memory Interface Specification Release 2.0",
 *     December 1, 2001
 * [2] Intel Corporation, "Intel PXA250 and PXA210 Application Processors
 *     Developer's Manual", February 2002, Order Number: 278522-001
 * [3] Intel Corporation, "Common Flash Interface (CFI) and Command Sets
 *     Application Note 646", April 2000, Order Number: 292204-004
 * [4] Advanced Micro Devices, "Common Flash Memory Interface Publication 100 Vendor & Device
 *     ID Code Assignments", December 1, 2001, Volume Number: 96.1
 *
 */

#include <stdint.h>
#include <stdlib.h>
#include <string.h>
#include <flash/cfi.h>
#include <flash/intel.h>

#include "part.h"
#include "bus.h"

int bcm1250_ejtag_do(parts *, uint64_t, uint64_t, int, int, unsigned char *, int);

int flash_erase_block( parts *ps, uint32_t adr );
int flash_unlock_block( parts *ps, uint32_t adr );
int flash_program( parts *ps, uint32_t adr, uint32_t data );
int flash_erase_block32( parts *ps, uint32_t adr );
int flash_unlock_block32( parts *ps, uint32_t adr );
int flash_program32( parts *ps, uint32_t adr, uint32_t data );

cfi_query_structure_t *detect_cfi( parts *ps, int o);
cfi_query_structure_t *cfi;

int blockstart(cfi_query_structure_t *, uint32_t);

void
flashmsbin( parts *ps, FILE *f )
{
	part *p = ps->parts[0];
	int o = 0;
	uint32_t adr;

	printf( "Note: Supported configuration is 2 x 16 bit only\n" );

	switch (bus_width( ps )) {
		case 16:
			o = 1;
			break;
		case 32:
			o = 2;
			break;
		default:
			printf( "Error: Unknown bus width!\n" );
			return;
	}

	/* EXTEST */
	part_set_instruction( p, "EXTEST" );
	parts_shift_instructions( ps );

	cfi = detect_cfi( ps, o );
	
	/* test sync bytes */
	{
		char sync[8];
		fread( &sync, sizeof (char), 7, f );
		sync[7] = '\0';
		if (strcmp( "B000FF\n", sync ) != 0) {
			printf( "Invalid sync sequence!\n" );
			return;
		}
	}

	/* erase memory blocks */
	{
		uint32_t start;
		uint32_t len;
		int first, last;

		fread( &start, sizeof start, 1, f );
		fread( &len, sizeof len, 1, f );
		first = start / (cfi->device_geometry.erase_block_regions[0].erase_block_size * 2);
		last = (start + len - 1) / (cfi->device_geometry.erase_block_regions[0].erase_block_size * 2);
		for (; first <= last; first++) {
			adr = first * cfi->device_geometry.erase_block_regions[0].erase_block_size * 2;
			flash_unlock_block32( ps, adr );
			printf( "block %d unlocked\n", first );
			printf( "erasing block %d: %d\n", first, flash_erase_block32( ps, adr ) );
		}
	}

	printf( "program:\n" );
	for (;;) {
		uint32_t a, l, c;

		fread( &a, sizeof a, 1, f );
		fread( &l, sizeof l, 1, f );
		fread( &c, sizeof c, 1, f );
		if (feof( f )) {
			printf( "Error: premature end of file\n" );
			return;
		}
		printf( "record: start = 0x%08X, len = 0x%08X, checksum = 0x%08X\n", a, l, c );
		if ((a == 0) && (c == 0))
			break;
		if (l & 3) {
			printf( "Error: Invalid record length!\n" );
			return;
		}

		while (l) {
			uint32_t data;

			printf( "addr: 0x%08X\r", a );
			fread( &data, sizeof data, 1, f );
			if (flash_program32( ps, a, data )) {
				printf( "\nflash error\n" );
				return;
			}
			a += 4;
			l -= 4;
		}
	}
	printf( "\n" );

	/* Read Array */
	bus_write( ps, 0 << o, 0x00FF00FF );

	fseek( f, 15, SEEK_SET );
	printf( "verify:\n" );

	for (;;) {
		uint32_t a, l, c;

		fread( &a, sizeof a, 1, f );
		fread( &l, sizeof l, 1, f );
		fread( &c, sizeof c, 1, f );
		if (feof( f )) {
			printf( "Error: premature end of file\n" );
			return;
		}
		printf( "record: start = 0x%08X, len = 0x%08X, checksum = 0x%08X\n", a, l, c );
		if ((a == 0) && (c == 0))
			break;
		if (l & 3) {
			printf( "Error: Invalid record length!\n" );
			return;
		}

		while (l) {
			uint32_t data, readed;

			printf( "addr: 0x%08X\r", a );
			fread( &data, sizeof data, 1, f );
			readed = bus_read( ps, a );
			if (data != readed) {
				printf( "\nverify error: 0x%08X vs. 0x%08X\n", readed, data );
				return;
			}
			a += 4;
			l -= 4;
		}
	}
	printf( "\n" );

	printf( "Done.\n" );
}

void
flashmem( parts *ps, FILE *f, uint32_t addr )
{
	part *p = ps->parts[0];
	int bw, ds;
	int o = 1;
	uint32_t adr;

	bw=bus_width(ps);

	ds=bw/8;

	/* EXTEST */
	part_set_instruction( p, "EXTEST" );
	parts_shift_instructions( ps );

	cfi = detect_cfi( ps , o );
	
	if(!cfi){
		printf("No flash detected.\n");
		return;
	}
	
	printf( "program:\n" );
	adr = addr;
#ifdef READONEBYTE
	while (!feof( f )) {
		uint32_t data=0;

		fprintf( stderr, "addr: 0x%08X   %i\r", adr, adr-addr );
		if(fread( &data, ds, 1, f )) {
			if (blockstart(cfi, adr)) {
				printf("Erasing block at 0x%08x\n",adr);
				flash_erase_block(ps, adr);
			}
			if (flash_program( ps, adr, data )) {
				printf( "\nflash error\n" );
				return;
			}
		}
		adr += ds;
	}
#else 
	while (!feof( f )) {
		unsigned char buf[32];
		int num;
		fprintf( stderr, "addr: 0x%08X   %i\r", adr, adr-addr );
		if((num=fread( buf, 1, 32, f ))>0) {
			int i;
			if (blockstart(cfi, adr)) {
				printf("Erasing block at 0x%08x\n",adr);
				flash_erase_block(ps, adr);
			}
			bus_write( ps, adr, 0xe8);
			bus_write( ps, adr, 0x1f);
//			for(i=0;i<32;i++) bus_write(ps, adr+i, buf[i]);
			bcm1250_ejtag_do(ps, 0x1fc00000+adr, 0, 0, 5, buf, 0);
			bus_write( ps, adr, 0xd0);
		}
		adr+=32;
	}
#endif
	if(cfi->identification_string.pri_id_code==1) {
		bus_write(ps, 0, 0xff);
	}
	printf( "\n" );
	fseek( f, 0, SEEK_SET );
	printf( "verify:\n" );
	adr = addr;
#ifdef READONEBYTE
	while (!feof( f )) {
		uint32_t data=0;
		uint32_t readed;
		fprintf(stderr, "addr: 0x%08X   %i\r", adr, adr-addr );
		if(fread( &data, ds, 1, f )) {
			readed = bus_read( ps, adr );
			if (data != readed) {
				fprintf(stderr, "\nverify error: 0x%08X vs. 0x%08X\n", readed, data );
				return;
			}
		}
		adr += ds;
	}
#else 
	while (!feof( f )) {
		int r;
		unsigned char buf[32], pbuf[32];
		fprintf(stderr, "addr: 0x%08X   %i\r", adr, adr-addr );
		if((r=fread( buf, ds, 32, f ))) {
			bcm1250_ejtag_do(ps, 0x1fc00000+adr, 0, 1, 5, pbuf, 0);
			if (memcmp(buf, pbuf, r)) {
				fprintf(stderr, "\nverify error\n");
				return;
			}
		}
		adr += r;
	}
	fprintf(stderr, "addr: 0x%08X   %i\r", adr, adr-addr );
#endif
	printf( "\nDone.\n" );
}

#define	CFI_INTEL_ERROR_UNKNOWN				1
#define	CFI_INTEL_ERROR_UNSUPPORTED			2
#define	CFI_INTEL_ERROR_LOW_VPEN			3
#define	CFI_INTEL_ERROR_BLOCK_LOCKED			4
#define	CFI_INTEL_ERROR_INVALID_COMMAND_SEQUENCE	5

int
flash_erase_block( parts *ps, uint32_t adr )
{
	int i;
	uint32_t o3;
	
	switch(cfi->identification_string.pri_id_code) {
		case 2:
			bus_write( ps, 0xaaa, 0xaa);
			bus_write( ps, 0x555, 0x55);
			bus_write( ps, 0xaaa, 0x80);
			bus_write( ps, 0xaaa, 0xaa);
			bus_write( ps, 0x555, 0x55);
			bus_write( ps, adr, 0x30);

			i=0;
			while(i<1000) {
				o3=bus_read( ps, adr);
				if((o3&0xff)==0xff) {
					printf("Erase done after %i reads.\n",i);	
					return 0;
				};
				i++;
			}
			fprintf(stderr, "Flash erase timeout\n");
			break;
		case 1:
			bus_write( ps, adr, 0x20);
			bus_write( ps, adr, 0xd0);
			i=0;
			while(i<50000) {
				o3=bus_read( ps, adr);
				if((o3&0x80)==0x80) {
					printf("Erase done after %i reads.\n",i);	
					bus_write( ps, 0, 0xff); /* return to READ ARRAY */
					return 0;
				};
				i++;
			}
			fprintf(stderr, "Flash erase timeout\n");
			bus_write( ps, 0, 0xff);
			bus_write( ps, 0, 0x50); /* clear status reg */
			bus_write( ps, 0, 0xff);
			
			break;
	}		
			
	return 0;
}

int
flash_unlock_block( parts *ps, uint32_t adr )
{
return 0;
}

int
flash_program( parts *ps, uint32_t adr, uint32_t data )
{

	switch(cfi->identification_string.pri_id_code) {
		case 2:
			bus_write( ps, 0xaaa, 0xaa);
			bus_write( ps, 0x555, 0x55);
			bus_write( ps, 0xaaa, 0xa0);
			bus_write( ps, adr, data);
			break;
		case 1:
			bus_write(ps, adr, 0x40);
			bus_write(ps, adr, data);
			break;
	}
	return 0;
}

int
flash_erase_block32( parts *ps, uint32_t adr )
{
return 0;
}

int
flash_unlock_block32( parts *ps, uint32_t adr )
{
return 0;
}

int
flash_program32( parts *ps, uint32_t adr, uint32_t data )
{
return 0;
}
