#ifdef RSOUND
/* sound.c: Sound support
   Copyright (c) 1999-2000 Philip Kendall

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

*/

/* XXX quick-hack, slightly crude Linux-only 48k sound support -rjm */

/* XXX ok, also had a fair crack at 128 :-), I'm not too sure about
 * the envelope stuff - that may be fairly broken.
 * -rjm
 */

/* some AY details (volume levels, white noise RNG algorithm) based on
 * info from MAME's ay8910.c - MAME's licence explicitly permits free
 * use of info (even encourages it).
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <sys/soundcard.h>

#include "sound.h"
#include "spectrum.h"


int sound_freq=16000;


#define AY_CLOCK		1773400
#define AY_FRAME_CLOCK		(AY_CLOCK/50)

/* assume all three tone channels together match the beeper volume.
 * Must be <=127 for all channels; 4 x 31 = 124.
 */
#define AMPL_BEEPER		31
#define AMPL_AY_TONE		31	/* three of these */

/* using 256 byte frags for 8kHz, scale up for higher */
#define BASE_SOUND_FRAG_PWR	8

static int sound_enabled=1;
static int soundfd=-1;
static int sound_framesiz;

static unsigned char ay_tone_levels[16];

static unsigned char *sound_buf,*sound_ptr;
static int sound_oldpos,sound_oldval;

static double ay_tone_tick[3],ay_noise_tick,ay_env_tick;
static double ay_tick_incr;
static int ay_tone_period[3],ay_noise_period,ay_env_period;
static int env_held=0,env_alternating=0;



void sound_init(void)
{
int frag,tmp,f;
double v;

if(!sound_enabled) return;

sound_framesiz=sound_freq/50;

if((sound_buf=malloc(sound_framesiz))==NULL)
  {
  sound_enabled=0;
  return;
  }

sound_oldval=128;
sound_oldpos=0;
sound_ptr=sound_buf;

if((soundfd=open("/dev/dsp",O_WRONLY/* |O_NONBLOCK */))<0)
  {
  free(sound_buf);
  sound_enabled=0;
  return;
  }

frag=(0x20000|BASE_SOUND_FRAG_PWR);

tmp=AFMT_S8;
ioctl(soundfd, SNDCTL_DSP_SETFMT, &tmp);

tmp=sound_freq;
if(ioctl(soundfd,SNDCTL_DSP_SPEED,&tmp)<0)
  {
  free(sound_buf);
  sound_enabled=0;
  return;
  }

/* XXX would need changing for >16kHz */
if(sound_freq>8000) frag++;

/* XXX check for err */
ioctl(soundfd,SNDCTL_DSP_SETFRAGMENT,&frag);

/* logarithmic volume levels, 3dB per step */
v=AMPL_AY_TONE;
for(f=15;f>0;f--)
  {
  ay_tone_levels[f]=(int)(v+0.5);
  /* 10^3/20 = 3dB */
  v/=1.4125375446;
  }
ay_tone_levels[0]=0;

ay_noise_tick=ay_noise_period=0;
ay_env_tick=ay_env_period=0;
for(f=0;f<3;f++)
  ay_tone_tick[f]=ay_tone_period[f]=0;

ay_tick_incr=(double)AY_CLOCK/sound_freq;
}


void sound_end(void)
{
if(sound_enabled)
  {
  free(sound_buf);
  close(soundfd);
  }
}


static void overlay_tone(int ofs,int chan,int level)
{
int f;

if(level && ay_tone_tick[chan]>=ay_tone_period[chan])
  sound_buf[ofs]+=level;
else
  sound_buf[ofs]-=level;

if(ay_tone_tick[chan]>=ay_tone_period[chan]*2)
  ay_tone_tick[chan]-=ay_tone_period[chan]*2;

ay_tone_tick[chan]+=ay_tick_incr;
}


static void overlay_ay(void)
{
static int rng=1;
static int noise_toggle=1;
static int env_level=0;
int tone_level[3];
int mixer=machine.ay.registers[7];
int envshape=machine.ay.registers[13];
int f,g;
int v=0;

if(!machine.ay.present) return;

/* the tone level if no enveloping is being used */
for(f=0;f<3;f++)
  tone_level[f]=ay_tone_levels[machine.ay.registers[8+f]&15];

for(f=0;f<sound_framesiz;f++)
  {
  /* envelope */
  if(ay_env_period)
    {
    if(!env_held)
      {
      v=((int)ay_env_tick*15)/ay_env_period;
      if(v<0) v=0;
      if(v>15) v=15;
      if((envshape&4)==0) v=15-v;
      if(env_alternating) v=15-v;
      env_level=ay_tone_levels[v];
      }
  
    for(g=0;g<3;g++)
      if(machine.ay.registers[8+g]&16)
        tone_level[g]=env_level;
  
    ay_env_tick+=ay_tick_incr;
    if(ay_env_tick>=ay_env_period)
      {
      ay_env_tick-=ay_env_period;
      if(!env_held && ((envshape&1) || (envshape&8)==0))
        {
        env_held=1;
        if((envshape&2) || (envshape&0xc)==4)
          env_level=ay_tone_levels[15-v];
        }
      if(!env_held && (envshape&2))
        env_alternating=!env_alternating;
      }
    }

  /* generate tone+noise */
  if((mixer&1)==0 || (mixer&0x08)==0)
    overlay_tone(f,0,(noise_toggle || (mixer&0x08))?tone_level[0]:0);
  if((mixer&2)==0 || (mixer&0x10)==0)
    overlay_tone(f,1,(noise_toggle || (mixer&0x10))?tone_level[1]:0);
  if((mixer&4)==0 || (mixer&0x20)==0)
    overlay_tone(f,2,(noise_toggle || (mixer&0x20))?tone_level[2]:0);

  /* update noise RNG/filter */
  ay_noise_tick+=ay_tick_incr;
  if(ay_noise_tick>=ay_noise_period)
    {
    if((rng&1)^((rng&2)?1:0))
      noise_toggle=!noise_toggle;
    
    /* rng is 17-bit shift reg, bit 0 is output.
     * input is bit 0 xor bit 2.
     */
    rng|=((rng&1)^((rng&4)?1:0))?0x20000:0;
    rng>>=1;
    
    ay_noise_tick-=ay_noise_period;
    }
  }
}


/* AY register itself will have already been updated, but we
 * may need to fiddle with timers etc.
 */
void sound_ay_write(int reg,int val)
{
switch(reg)
  {
  case 0: case 1: case 2: case 3: case 4: case 5:
    ay_env_tick=0.;
    env_held=env_alternating=0;
    ay_tone_tick[reg>>1]=0.;
    ay_tone_period[reg>>1]=8*(machine.ay.registers[reg&~1]|
                              (machine.ay.registers[reg|1]&15)<<8);
    break;
  case 6:
    ay_noise_tick=0.;
    ay_noise_period=16*(machine.ay.registers[reg]&31);
    break;
  case 11: case 12:
    ay_env_tick=0.;
    env_held=env_alternating=0;
    ay_env_period=256*(machine.ay.registers[11]|
                       (machine.ay.registers[12])<<8);
    break;
    
    /* XXX any others? */
  }
}


void sound_frame(void)
{
int f, i, s;

if(!sound_enabled) return;

for(f=sound_oldpos;f<sound_framesiz;f++)
  sound_buf[f]=sound_oldval;

//overlay_ay();

i=0;
s=0;
while(s<sound_framesiz) {
    i=write(soundfd,sound_buf+s,sound_framesiz-s);
    if(i>0) {
        s+=i;
    }
    
}

sound_oldpos=0;
sound_ptr=sound_buf;
}


void sound_beeper(int on)
{
int newpos;
int f;

on=(on?128+AMPL_BEEPER:128-AMPL_BEEPER);

if(on==sound_oldval) return;

/* XXX faster way? */
newpos=(tstates*sound_freq)/machine.hz;

if(newpos>=0)
  for(f=sound_oldpos;f<=newpos && f<sound_framesiz;f++)
    sound_buf[f]=sound_oldval;

sound_oldpos=newpos;
sound_oldval=on;
}
#endif
