/* 
   fstv - display TV picture and control the TV tuner

   Copyright (C) 1997 Ralph Metzler (rjkm@thp.uni-koeln.de)
   Copyright (C) 1998 Heikki Hannikainen <hessu@pspt.fi>
   Copyright (C) 1999 Christian Gebauer (gebauer@bigfoot.com)
   Copyright (C) 1999 Matan Ziv-Av <zivav@cs.bgu.ac.il>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   
   Changelog:
   	Initial version by Ralph Metzler (rjkm@thp.uni-koeln.de)
   	sometime during the year 1997...
   
   	31-Jan-1998	Heikki Hannikainen <hessu@pspt.fi>:

   		- Reformatted (indent...)
   		- Split up the stuff to functions, comment it to some degree
   		  (just to make myself clear how it works...)
   		- Added support for xtvscreen's config file ~/.xtvrc
   		  for input of channel data
   		- Added commands for selecting channels
   		- 7-segment LED styled channel indicator (ugly code)

	23-Jan-1999 Christian Gebauer (gebauer@bigfoot.com):

		- Removed the channel indicator (it is not very useful
		  if you want to use your computer monitor as a tv-set).
		- Added proper initialization of the bttv-driver.
		- Added support for the picture-parameters in
		  xtvscreens config file.
		- Added support for different video-sources by pressing 'V'
		- sound card support:
		  glidetv is now able to adjust the audio volume.
			On exit the original volume is restored
		  The mayor advantage is that glidetv avoids the annoying
			cracking sound during the channel-switch by muting the audio
			during the procedure.
		- Add a key for muting audio.
		- Added a "Hot-key". With '/' the current channel is saved,
		  the '*'-Key recalls it latter. This is useful for switching
			to a channel with a high number.
		- Added more "channel-keys" in the row "asdfghjkl" (Channel
		  Number 10 to 20)
		- Fixed the weird reaction on function-keys.
		- Tweaked the grSstWinOpen-call a little bit, because the old
		  one doesn't work with a newer glide on my machine.

        15-feb-1999 Matan Ziv-Av
           	- Added support for svgalib.
                - Both glide and svgalib support are compile time options, as
                  well as run time selectable (if both are compiled in).
                - Controls for contrast, brightness, hue, colour.
                - Fine tuning
                - Changing Channels
                - VC switch handling
                - Kernel framebuffer support.
TODO:
   document controls
*/

#define CFGLEN 256

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <ctype.h>
#include <linux/soundcard.h>

#ifdef GLIDE
#include <glide.h>
#include <linutil.h>
#include <sys/vt.h>
#include <signal.h>
#define TARGET_GLIDE 2
#endif

#ifdef SVGALIB
#include <vga.h>
#define TARGET_SVGALIB 1
#endif

#ifdef FB
#include <linux/fb.h>
#include <sys/mman.h>
#include <linux/kd.h>
#define TARGET_FB 3
#endif

#ifdef JPEG
#include <jpeglib.h>
#endif

#include <linux/videodev.h>  /* make sure its the one from the bttv version
   				you use, if you don't use the kernel bttv */ 
#include "keys.h"
#include "channels.h"

struct channel_t 
{
  char *name;
  int freq;
  int nr;
  int contrast;
  int brightness;
  int hue;
  int colour;
  int volume;
  int finetune;
  
  struct channel_t *next;
  struct channel_t *prev;
} *channels = NULL;

char	*bttvname = "/dev/bttv";	/* BTTV device name */
int	fbttv;				/* File descriptor for the device */
int one=1;
int zero=0;
int bttv_up=0;                      
struct video_capability vcap; 
struct video_channel *vchan;
int cur_source=0;

int target;
int capturing = 1;
struct channel_t *channel;
	
/* Real video resolution used: */
int width = 768, height = 576;		 /* set this to 640 x 480 for NTSC */

/* Glide graphics: */
#ifdef GLIDE
GrHwConfiguration hwconfig;	      	/* Hardware configuration */
GrLfbInfo_t info;
int glide_closed=1;
int tty_fd;
struct vt_mode old_tty_mode;
#endif
#ifdef SVGALIB
vga_modeinfo *modeinfo;
#endif
#ifdef FB
char	*fbname = "/dev/fb0";		/* FB device name */
int	ffb;
#ifndef GLIDE
int tty_fd;
struct vt_mode old_tty_mode;
#endif
#endif

#ifdef JPEG
int jpeg_quality = 85 ;
#endif

char *graphmem;
int linewidth;
int bpp;		/* bytes per pixel */
int bitspp;
int verbose=0;

/* Mixer */
int mix_fd=-1;
int vol_up;
int vol_main;
int tuner_vol;
int vol_local_orig, vol_global_orig;
int mix_local_id;
int mix_global_id;
#define LEFT  0x0001
#define RIGHT 0x0100
int audio_mute = 0;
int audio_disabled = 1;
#define VOL_GLOBAL 1
#define VOL_LOCAL 2
#define VOL_TUNER 3
#define VOL_MAIN 4 
/*
 *  Print error-message and try close glide
 */
#ifdef GLIDE
void close_glide(void);
#endif
#ifdef SVGALIB
void close_svgalib(void);
#endif
#ifdef FB
void close_fb(void);
#endif
void toggle_capturing(int sc_mode);

void tv_getscansegment(char *ptr, int x, int y, int bytes)
{
   switch(target) {
#ifdef GLIDE
   	case TARGET_GLIDE:
   	     memcpy(ptr,graphmem+linewidth*y+bpp*x,bytes);	
           break;
#endif
#ifdef SVGALIB
        case TARGET_SVGALIB:
 	     memcpy(ptr,graphmem+linewidth*y+bpp*x,bytes);	
/*           vga_getscansegment(ptr,x,y,bytes);*/
           break;
#endif
#ifdef FB
   	case TARGET_FB:
   	     memcpy(ptr,graphmem+linewidth*y+bpp*x,bytes);	
           break;
#endif
   };
};

writeppm(FILE *file)
{
   int y;
   unsigned char tmp[8192], tmp2[8192];
   int i,j,k;

   fprintf(file,"P6\n%i %i\n255\n",width,height);
   for(y=0;y<height;y++){
      tv_getscansegment(tmp,0,y,width*bpp);
      switch(bitspp){
          case 16:
             for(i=0;i<width;i++){
                j=tmp[i*2]+256*tmp[i*2+1];
                tmp2[i*3]=(j&0xf800)>>8;
                tmp2[i*3+1]=(j&0x7e0)>>3;
                tmp2[i*3+2]=(j&0x1f)<<3;
             };
             break;
          case 24:
             memcpy(tmp2,tmp,width*3);
             break;
      };
      fwrite(tmp2,width*3,1,file);
   };
};   

#ifdef JPEG
writejpeg(FILE *file)
{
   int y;
   unsigned char tmp[4096], *tmp2, *tmp3;
   int i,j,k;
   struct jpeg_compress_struct cinfo;
    struct jpeg_error_mgr jerr;
    JSAMPROW row_pointer[1];

   tmp2=tmp3=malloc(width*height*3);
   for(y=0;y<height;y++){
      tv_getscansegment(tmp,0,y,width*bpp);
      switch(bitspp){
          case 16:
             for(i=0;i<width;i++){
                j=tmp[i*2]+256*tmp[i*2+1];
                tmp2[i*3]=(j&0xf800)>>8;
                tmp2[i*3+1]=(j&0x7e0)>>3;
                tmp2[i*3+2]=(j&0x1f)<<3;
             };
             break;
          case 24:
             memcpy(tmp2,tmp,width*3);
             break;
      };
      tmp2+=width*3;
   };
   
   cinfo.err = jpeg_std_error(&jerr);
   jpeg_create_compress(&cinfo);
 
   jpeg_stdio_dest(&cinfo, file);

   cinfo.image_width = width; 	
   cinfo.image_height = height;
   cinfo.input_components = 3;		
   cinfo.in_color_space = JCS_RGB; 	
   
   jpeg_set_defaults(&cinfo);
   jpeg_set_quality(&cinfo, jpeg_quality, TRUE );

   jpeg_start_compress(&cinfo, TRUE);

   while (cinfo.next_scanline < cinfo.image_height) {
      row_pointer[0] = & tmp3[cinfo.next_scanline * width*3];
      jpeg_write_scanlines(&cinfo, row_pointer, 1);
   }

   jpeg_finish_compress(&cinfo);
   jpeg_destroy_compress(&cinfo);
   free(tmp3);
};   
#endif

writeraw(FILE *file)
{
   int y;
   char tmp[4096], tmp2[8096];
   
/*   for(y=0;y<600;y++){
      tv_getscansegment(tmp,0,y,800*bpp);
      fwrite(tmp,800*bpp,1,file);
   };*/
   fwrite(graphmem,800*600*bpp,1,file);
};   

void fail(char *msg)
{
  if (bttv_up)
    {
      ioctl(fbttv, VIDIOCCAPTURE,&zero);
      close(fbttv);
    }
  printf(msg);
  switch(target) {
#ifdef GLIDE
     case TARGET_GLIDE:
        close_glide();
        break;
#endif
#ifdef SVGALIB
     case TARGET_SVGALIB:
        close_svgalib();
        break;
#endif
#ifdef FB
     case TARGET_FB:
        close_fb();
        break;
#endif
  };
  exit(1);
}

void open_mixer(void)
{
   if (!audio_disabled) {
      mix_fd=open("/dev/mixer",O_RDWR);
      if(mix_fd<0) {
         printf("Fail: open /dev/mixer/\n");
         exit(1);
      }
      if(ioctl(mix_fd,MIXER_READ(mix_global_id),&vol_global_orig)<0) {
         printf("Fail: get volume\n");
         exit(1);
      }
      if(ioctl(mix_fd,MIXER_READ(mix_local_id),&vol_local_orig)<0) {
         printf("Fail: get volume\n");
         exit(1);
      }
   }
}

void close_mixer(void)
{
	if (!audio_disabled) {
		if(ioctl(mix_fd,MIXER_WRITE(mix_local_id),&vol_local_orig)<0)
			fail("Fail: set volume\n");	
		if(ioctl(mix_fd,MIXER_WRITE(mix_global_id),&vol_global_orig)<0)
			fail("Fail: set volume\n");	
		close(mix_fd);
	}
}

#ifdef GLIDE
void (*glide_old_gotoback)();
void (*glide_old_comefromback)();

void glide_comefromback(int n);
void glide_gotoback(int n)
{
   	toggle_capturing(0);
        ioctl(tty_fd,VT_RELDISP,1);
        signal(SIGUSR1,glide_gotoback);
        signal(SIGUSR2,glide_comefromback);
};

void glide_comefromback(int n)
{
        ioctl(tty_fd,VT_RELDISP, VT_ACKACQ);
   	toggle_capturing(0);
        signal(SIGUSR1,glide_gotoback);
        signal(SIGUSR2,glide_comefromback);
};
void init_glide(void)
{
  GrScreenResolution_t resolution = GR_RESOLUTION_800x600;
  
  float scrWidth = 800.0f;
  float scrHeight = 600.0f;
  struct vt_mode tty_mode;
  
  if(glide_closed){
        grGlideInit();
        assert(grSstQueryHardware(&hwconfig));
        grSstSelect(0);
        assert(grSstWinOpen(0, resolution, GR_REFRESH_60Hz, GR_COLORFORMAT_ABGR, GR_ORIGIN_UPPER_LEFT, 2, 0));
        grDitherMode(GR_DITHER_DISABLE);
        grClipWindow(0, 0, (FxU32) scrWidth, (FxU32) scrHeight);
        grBufferClear(0x00, 0, 0);
    
        info.size = sizeof(GrLfbInfo_t);
    
        if (grLfbLock(GR_LFB_WRITE_ONLY, GR_BUFFER_FRONTBUFFER,
	            GR_LFBWRITEMODE_565, GR_ORIGIN_UPPER_LEFT, FXFALSE, &info)
	        == FXFALSE) 
        {
          fprintf(stderr, "grLfbLock: Failed to take write lock\n");
          grGlideShutdown();
          exit(1);
        }    
        graphmem=info.lfbPtr;
        linewidth=info.strideInBytes;
        bpp=2;
        bitspp=16;
        glide_closed=0;
        tty_fd=open("/dev/console",O_RDWR);
        ioctl(tty_fd,VT_GETMODE,&old_tty_mode);
        tty_mode=old_tty_mode;
        tty_mode.mode=VT_PROCESS;
        tty_mode.relsig=SIGUSR1;
        tty_mode.acqsig=SIGUSR2;
        ioctl(tty_fd,VT_SETMODE,&tty_mode);
        close(tty_fd);
        glide_old_gotoback=signal(SIGUSR1,glide_gotoback);
        glide_old_comefromback=signal(SIGUSR2,glide_comefromback);
  };
}

void close_glide(void)
{
  grLfbUnlock(GR_LFB_WRITE_ONLY, GR_BUFFER_FRONTBUFFER);
  grGlideShutdown();
  signal(SIGUSR1,glide_old_gotoback);
  signal(SIGUSR2,glide_old_comefromback);
  glide_closed=1;
}
#endif

#ifdef SVGALIB
void svga_gotoback(void)
{
toggle_capturing(0);
};

void svga_comefromback(void)
{
toggle_capturing(0);
};

void init_svgalib(void)
{
  int mode = G800x600x64K;
  
  float scrWidth = 800.0f;
  float scrHeight = 600.0f;
  
  vga_init();
	
  vga_setmode(mode);
  vga_setlinearaddressing();

  modeinfo = vga_getmodeinfo(mode);
  graphmem= vga_getgraphmem();
  linewidth=modeinfo->linewidth;
  bpp=2;
  bitspp=16;
#ifdef VGA_GOTOBACK
  vga_runinbackground(VGA_GOTOBACK,svga_gotoback);
  vga_runinbackground(VGA_COMEFROMBACK,svga_comefromback);
#endif
    
}

void close_svgalib(void)
{
   vga_setmode(TEXT);
}
#endif

#ifdef FB
void (*fb_old_gotoback)();
void (*fb_old_comefromback)();

unsigned char *save_fb;
int save_size;

void fb_comefromback(int n);
void fb_gotoback(int n)
{
   	toggle_capturing(0);
        ioctl(tty_fd,VT_RELDISP,1);
        signal(SIGUSR1,glide_gotoback);
        signal(SIGUSR2,glide_comefromback);
};

void fb_comefromback(int n)
{
        ioctl(tty_fd,VT_RELDISP, VT_ACKACQ);
   	toggle_capturing(0);
        signal(SIGUSR1,glide_gotoback);
        signal(SIGUSR2,glide_comefromback);
};

void init_fb(void)
{
  GrScreenResolution_t resolution = GR_RESOLUTION_800x600;
  
  float scrWidth = 800.0f;
  float scrHeight = 600.0f;
  struct vt_mode tty_mode;
  struct fb_fix_screeninfo fix_info;
  struct fb_var_screeninfo var_info;
  unsigned int offset;
  int mem_fd;
  
  if((ffb=open(fbname,O_RDWR))==-1)
      fail("Error openning framebuffer device \n");

  if(ioctl(ffb,FBIOGET_FSCREENINFO,&fix_info)==-1)
      fail("FBIOGET_FSCREENINFO failed \n");

  if(ioctl(ffb,FBIOGET_VSCREENINFO,&var_info)==-1)
      fail("FBIOGET_VSCREENINFO failed \n");
  
  if(var_info.bits_per_pixel<15)
      fail("Colour depth must be at least 15 bits per pixel \n");
  
  if((var_info.xres_virtual<width)||(var_info.yres_virtual<height))
      fail("Virtual screen not big enough \n");

  bitspp=var_info.bits_per_pixel;
  bpp=(bitspp+7)>>3;
  
  offset=(unsigned int)fix_info.smem_start;
  if((mem_fd=open("/dev/mem",O_RDWR))==-1)
      fail("Error openning framebuffer device \n");
  if((graphmem=mmap(0,fix_info.smem_len,PROT_READ|PROT_WRITE,MAP_SHARED,mem_fd,offset))==NULL)
      fail("Error in mmap\n");
  close(mem_fd);
  
  if((save_fb=malloc(fix_info.smem_len))==NULL)
      fail("malloc error.\n");

  save_size=fix_info.smem_len;
  memcpy(save_fb,graphmem,fix_info.smem_len);
  
  linewidth=fix_info.line_length;

  tty_fd=open("/dev/console",O_RDWR);
  ioctl(tty_fd,VT_GETMODE,&old_tty_mode);
  tty_mode=old_tty_mode;
  tty_mode.mode=VT_PROCESS;
  tty_mode.relsig=SIGUSR1;
  tty_mode.acqsig=SIGUSR2;
  ioctl(tty_fd,VT_SETMODE,&tty_mode);
  ioctl(tty_fd,KDSETMODE, KD_GRAPHICS);
  fb_old_gotoback=signal(SIGUSR1,glide_gotoback);
  fb_old_comefromback=signal(SIGUSR2,glide_comefromback);
        
}

void close_fb(void)
{
   ioctl(tty_fd,KDSETMODE, KD_TEXT);
   close(tty_fd);
   memcpy(graphmem,save_fb,save_size);
   free(save_fb);
   /* restore fb */
}
#endif

void open_bttv(void)
{
  if ((fbttv = open(bttvname, O_RDWR)) < 0) {
    fprintf(stderr, "Cannot open %s: %s\n", bttvname, strerror(errno));
    exit(1);
  }
  bttv_up =1;
}

void init_bttv(void)
{
  struct video_window vwin;
  struct video_buffer vbuf;
  struct video_audio *vaudio;
  int i;
  struct channel_t *channel;

  if(ioctl(fbttv,VIDIOCGCAP,&vcap))
    fail("VIDIOCGCAP failed \n");

  vchan = (struct video_channel*)malloc(sizeof(struct video_channel)*vcap.channels);
  memset(vchan,0,sizeof(struct video_channel)*vcap.channels);
  for (i = 0; i < vcap.channels; i++) {
		vchan[i].channel = i;
		if (ioctl(fbttv,VIDIOCGCHAN,&vchan[i]))
			fail("VIDIOCGCHAN failed \n");
	}

  vaudio = (struct video_audio*)malloc(sizeof(struct video_audio)*vcap.audios);
  memset(vaudio,0,sizeof(struct video_audio)*vcap.audios);
  for (i = 0; i < vcap.audios; i++) {
		vaudio[i].audio = i;
		if (-1 == ioctl(fbttv,VIDIOCGAUDIO,&vaudio[i]))
			fail("VIDIOCGAUDIO failed \n");
	}

  if(ioctl(fbttv, VIDIOCGWIN, &(vwin)) == -1)
    fail("VIDIOCGWIN failed \n");

  if(ioctl(fbttv, VIDIOCGFBUF, &(vbuf)) == -1)
    fail("VIDIOCGFBUF failed \n");

  vbuf.base=(void *)(1|(unsigned long)graphmem);
  vbuf.depth=bitspp;
  vbuf.bytesperline=linewidth;
  vbuf.width=800;
  vbuf.height=600;
  if(ioctl(fbttv, VIDIOCSFBUF, &vbuf))
    fail("VIDIOCSFBUF failed \n");
	
  vaudio[0].flags &= ~VIDEO_AUDIO_MUTE;
  tuner_vol=vaudio[0].volume;
  if (-1 == ioctl(fbttv,VIDIOCSAUDIO,&vaudio[0]))
    fail("VIDIOCSAUDIO failed \n");

  vwin.clipcount=0;
  vwin.flags=0;
  vwin.y=0;
  vwin.x=0;
  vwin.width=width;
  vwin.height=height;
  if (ioctl(fbttv, VIDIOCSWIN, &vwin))
    fail("VIDIOCSWIN failed \n");
	
  if ((cur_source >= vcap.channels) || (cur_source < 0))   /* Wrong commandline parameter */
	cur_source = 0;
  if(ioctl(fbttv, VIDIOCSCHAN,&vchan[cur_source]))
    fail("VIDIOCSCHAN failed \n");
	
  channel=channels;
  i = channel->freq;
  if(ioctl(fbttv, VIDIOCSFREQ,&i))
    fail("VIDIOCSFREQ failed \n");
   
  if(ioctl(fbttv, VIDIOCCAPTURE,&one))
    fail("VIDIOCCAPTURE failed \n");
  
  free(vaudio);
}

void close_bttv(void)
{
  free(vchan);
  if(ioctl(fbttv, VIDIOCCAPTURE,&zero))
    fail("VIDIOCCAPTURE failed \n");
  close(fbttv);
}

void get_fstvrc(char *path, int size)
{
  const char *home = getenv("HOME");
  const char *file = ".fstvrc";
	
  strncpy(path, home, size);
  strncat(path, "/", size);
  strncat(path, file, size);
}

void read_config(void)
{
  FILE *f;
  char fn[1024];
  char s[CFGLEN];
  char *cmd, *par;
  int i = 0;
  struct channel_t *channel = NULL, *prev, **prevp = &channels;
  
  get_fstvrc(fn, 1024);
	
  if (!(f = fopen(fn, "r"))) {
		fprintf(stderr, "Could not open %s for reading: %s\n", fn, strerror(errno));
		exit(1);
	}
  
  while (fgets(s, CFGLEN, f)) {
		cmd = strtok(s, " ");
		par = strtok(NULL, "\n");
		if (!strcasecmp(cmd, "Channel:")) {
			prev = channel;
			channel = malloc(sizeof(struct channel_t));
			*prevp = channel;
			prevp = &channel->next;
			channel->prev = prev;
			i++;
			channel->nr = i;
                        channel->volume=-1;
                        channel->finetune=0;
			channel->name = strdup(par);
		};
		if ((!strcasecmp(cmd, "Frequency:")) && (channel)) {
	             channel->freq = atoi(par);
	             if(verbose)fprintf(stderr, "\nChannel %d: %s at %d", channel->nr, channel->name, channel->freq);
	        }; 
		if ((!strcasecmp(cmd, "CBHC:")) && (channel)) {
                     sscanf(par,"%i%i%i%i",&(channel->contrast),
                     &(channel->brightness),&(channel->hue),&(channel->colour));
                     if(verbose)fprintf(stderr,"   (%s)",par);
                };
                if ((!strcasecmp(cmd, "Volume:")) && (channel)) {
                     sscanf(par,"%i",&(channel->volume));
                     if(verbose)fprintf(stderr,"   (%s)",par);
                };
                if ((!strcasecmp(cmd, "FT:")) && (channel)) {
                     sscanf(par,"%i",&(channel->finetune));
                     if(verbose)fprintf(stderr,"   (%s)",par);
                };  
	}
  
  if(verbose)fprintf(stderr,"\n");
  fclose(f);
  
  if (!channel) {
		channel = malloc(sizeof(struct channel_t));
                channels=channel;
                channel->freq=3476; /* Israel channel 1 in En-Yahav */
                channel->name=strdup("1");
                channel->finetune=0;
                channel->contrast=255;
                channel->brightness=0;
                channel->hue=0;
                channel->colour=255;
                channel->volume=80;
                channel->nr=1;
	};
  
  channel->next = channels;
  channel->next->prev = channel;
};

void write_config(void)
{
  FILE *f;
  char fn[1024];
  char s[CFGLEN];
  char *cmd, *par;
  int i = 0;
  struct channel_t *channel = NULL, *prev, **prevp = &channels;
  
  get_fstvrc(fn, 1024);
	
  if (!(f = fopen(fn, "w"))) {
		fprintf(stderr, "Could not open %s for writing: %s\n", fn, strerror(errno));
                return;
	}
  channel=channels;
  
  while (channel!=NULL) {
		fprintf(f,"\n*\nChannel: %s\n",channel->name);
		fprintf(f,"Frequency: %i\n",channel->freq);
		fprintf(f,"CBHC: %i %i %i %i\n",channel->contrast,
                        channel->brightness,channel->hue,channel->colour);
                if(channel->volume!=-1)
                   fprintf(f,"Volume: %i\n",channel->volume);
                if(channel->finetune!=-1)
                   fprintf(f,"FT: %i\n",channel->finetune);
                channel=channel->next;
                if(channel==channels)break;
        };
  
  fclose(f);
  
};

struct channel_t *find_channel(struct channel_t *old, int i)
{
  struct channel_t *ch = channels;
  
  do {
		ch = ch->next;
	} while ((ch->nr != i) && (ch != channels));
  
  if (ch->nr == i)
    return ch;
   else
    return old;
}

void renumber_channels(void)
{
  struct channel_t *ch = channels;
  int i;
  
  ch->nr=i=1;
  while(ch->next!=channels){
		ch = ch->next;
                ch->nr=++i;
  };  
}

void cmd_line(int argc, char **argv)
{
  int i = 1;
	
        mix_local_id=-1;
        target=-1;
        
	while (i < argc) {
		if (strcmp("-h",argv[i]) == 0) {
				printf("Usage: fstv [-h] [-v] [-n] [-s Number] [-a Channel Volume]\n\n");
				printf("       -h                 this message\n");
				printf("       -v                 verbose messages to stdout\n");
				printf("       -s Number          select the default videosource\n");		
				printf("       -a Channel Volume  enable soundcard support:\n");
				printf("                          Channel: {line/cd/mic}\n");
				printf("                          Volume: 0-100\n");
                                printf("       -n 		  Don't control the sound card.\n");
                                printf("       -t Target	  Select target:\n");
#ifdef GLIDE
                                printf("			  glide\n");
#endif
#ifdef SVGALIB
                                printf("			  svgalib\n");
#endif
#ifdef FB
                                printf("			  fb\n");
#endif
				exit(0);
		} 
                if (strcmp("-s",argv[i]) == 0) {
				cur_source=atoi(argv[i+1]);
				i+=2;
                                continue;
                } 
                if (strcmp("-v",argv[i]) == 0) {
				verbose=1;
				i+=1;
                                continue;
                } 
                if (strcmp("-a",argv[i]) == 0) {
				audio_disabled = 0;
				if (strcmp("line",argv[i+1]) == 0)
                                   	mix_local_id = SOUND_MIXER_LINE;
                                if (strcmp("cd",argv[i+1]) == 0)
					mix_local_id = SOUND_MIXER_CD;
                                if (strcmp("mic",argv[i+1]) == 0)
					mix_local_id = SOUND_MIXER_MIC;
                                if(mix_local_id==-1){
                                   	printf("Wrong audio-parameter\n");
					exit(1);
                                }
				vol_up = atoi(argv[i+2]);
				if ((vol_up > 100)||(vol_up <= 0)) {
					printf("Wrong audio-parameter\n");
					exit(1);						
                                }
				i+=3;
                                continue;
                }
		if (strcmp("-t",argv[i]) == 0) {
#ifdef GLIDE
                		if (strcmp("glide",argv[i+1]) == 0)
                                           	target = TARGET_GLIDE;
#endif
#ifdef SVGALIB
                                if (strcmp("svgalib",argv[i+1]) == 0)
                                   		target = TARGET_SVGALIB;
                                 
#endif
#ifdef FB
                                if (strcmp("fb",argv[i+1]) == 0)
                                   		target = TARGET_FB;
                                 
#endif
                                if(target==-1){
                                                printf("Wrong target parameter\n");
						exit(1);
                                }
				i+=2;
                                continue;
                }
                if (strcmp("-n",argv[i]) == 0) {
                                mix_local_id=-2;
                                i+=1;
                                continue;
                }
                i++;
	}
        if(mix_local_id==-1){
           	audio_disabled=0;
           	mix_local_id = SOUND_MIXER_LINE;
                vol_up=80;
        };	/* These defaults fit my computer */
        if(target==-1){
#ifdef SVGALIB
                target = TARGET_SVGALIB;
#else
#ifdef GLIDE
                target = TARGET_GLIDE;
#else
#ifdef FB
                target = TARGET_FB;
#else
                printf("No Target\n");
                exit(1);     
#endif
#endif
#endif           
        };	
        mix_global_id=SOUND_MIXER_VOLUME;           
        vol_main=100;
}

void init(int argc, char **argv)
{
	cmd_line(argc,argv);
  read_config();
  open_mixer();
  open_bttv();
  switch(target) {
#ifdef GLIDE
     case TARGET_GLIDE:
        init_glide();
        break;
#endif
#ifdef SVGALIB
     case TARGET_SVGALIB:
        init_svgalib();
        break;
#endif
#ifdef FB
     case TARGET_FB:
        init_fb();
        break;
#endif
  };
  init_bttv();
}

void closedown(void)
{
  close_mixer();
  close_bttv();
  switch(target) {
#ifdef GLIDE
     case TARGET_GLIDE:
        close_glide();
        break;
#endif
#ifdef SVGALIB
     case TARGET_SVGALIB:
        close_svgalib();
        break;
#endif
#ifdef FB
     case TARGET_FB:
        close_fb();
        break;
#endif
  };
  if(verbose)fprintf(stderr, "TV closed.\r\n");
  exit(0);
}

void set_volume(int control,int vol)
{
   int mix_id;
      
   if(control==VOL_TUNER) {
        struct video_audio vaudio;

	if (-1 == ioctl(fbttv,VIDIOCGAUDIO,&vaudio))
           fail("VIDIOCGAUDIO failed \n");
        
        vaudio.flags &= ~VIDEO_AUDIO_MUTE;
        vaudio.volume=vol;
        if (-1 == ioctl(fbttv,VIDIOCSAUDIO,&vaudio))
           fail("VIDIOCSAUDIO failed \n");
        
        return;
   };

   if(vol>100)vol=100;
   
   switch(control) {
      case VOL_LOCAL:
         mix_id=mix_local_id;
         break;
      case VOL_GLOBAL:
         mix_id=mix_global_id;
         break;
      default:
         mix_id=0;
   };
   if (!audio_disabled) {
		if (!audio_mute) {
			vol=(LEFT+RIGHT)*vol;
			if(ioctl(mix_fd,MIXER_WRITE(mix_id),&vol)<0)
				fail("Fail: set volume\n");
		}
	}
}


/*
 *	Change the tuner frequency
 */
void set_pict_params(struct channel_t *channel)
{
  struct video_picture params;
  if (ioctl(fbttv, VIDIOCGPICT, &params)) 
		fail("VIDIOCGPICT failed \n");
	else {
		params.contrast = channel->contrast<<7;
		params.brightness = (channel->brightness+128)<<8;
		params.hue = (channel->hue+128)<<8;
		params.colour = channel-> colour<<7;
		if (ioctl(fbttv, VIDIOCSPICT, &params))
			fail("VIDIOCSPICT failed \n");
	}
};
 
void change_channel(struct channel_t *channel)
{
  struct video_picture params;
  ulong i = channel->freq;
    
  if (!channel) {
		fprintf(stderr, "Cannot change to a NULL channel!\n");
		closedown();
	}
  
	set_volume(VOL_LOCAL,0);
	
  if(channel->volume==-1)channel->volume=80;      
  
  if(ioctl(fbttv, VIDIOCCAPTURE,&zero))
    fail("VIDIOCCAPTURE failed \n");

  if(verbose)fprintf(stderr, "Choosing channel %d: %s\r\n", channel->nr, channel->name);
  
  i+=channel->finetune;
  if (ioctl(fbttv, VIDIOCSFREQ, &i))
    fail("VIDIOCSFREQ failed \n");
	
	usleep(99999);

  if (ioctl(fbttv, VIDIOCGPICT, &params)) 
		fail("VIDIOCGPICT failed \n");
	else {
		params.contrast = channel->contrast<<7;
		params.brightness = (channel->brightness+128)<<8;
		params.hue = (channel->hue+128)<<8;
		params.colour = channel-> colour<<7;
		if (ioctl(fbttv, VIDIOCSPICT, &params))
			fail("VIDIOCSPICT failed \n");
	}
 
  if(ioctl(fbttv, VIDIOCCAPTURE,&one))
    fail("VIDIOCCAPTURE failed \n");

  set_volume(VOL_LOCAL,channel->volume);
}


void toggle_capturing(int sc_mode)
{	
	if (capturing) {
		if(ioctl(fbttv, VIDIOCCAPTURE,&zero))
			fail("VIDIOCCAPTURE failed \n");
#ifdef GLIDE
		if((target==TARGET_GLIDE)&&!sc_mode)close_glide();
#endif
		if(verbose)fprintf(stderr, "Capturing disabled, press C again to continue...\r\n");
		capturing = 0;
	} else {
#ifdef GLIDE
		if(target==TARGET_GLIDE)init_glide();
#endif
		if(ioctl(fbttv, VIDIOCCAPTURE,&one))
			fail("VIDIOCCAPTURE failed \n");
		if(verbose)fprintf(stderr, "Capturing enabled.\r\n");
		capturing = 1;
	}
}

void toggle_audio(void)
{
	if (!audio_disabled) {
		if (audio_mute) {
			audio_mute = 0;
			set_volume(VOL_LOCAL,channel->volume);
			if(verbose)fprintf(stderr, "Audio unmuted.\r\n");
		} else {
			set_volume(VOL_LOCAL,0);
			if(verbose)fprintf(stderr, "Audio muted, press M again to unmute...\r\n");
			audio_mute = 1;
		}
	}
}

void decrease_volume(int control)
{	
	if (control==VOL_TUNER) {
		if ((tuner_vol-=300)<0)
			tuner_vol = 0;
		set_volume(VOL_TUNER,tuner_vol);
		if(verbose)fprintf(stderr, "Tuner volume at %i.\r\n",tuner_vol);
                return;
        };
        if (!audio_disabled)
		if (control==VOL_GLOBAL) {
			if ((vol_up-=2)<0)
				vol_up = 0;
			set_volume(VOL_GLOBAL,vol_up);
			if(verbose)fprintf(stderr, "Audio volume at %i.\r\n",vol_up);
		}
		if (control==VOL_LOCAL) {
			if ((channel->volume-=2)<0)
                           channel->volume = 0;
			set_volume(VOL_LOCAL,(channel->volume*vol_main)/100);
			if(verbose)fprintf(stderr, "Channel volume at %i.\r\n",channel->volume);
		}
		if (control==VOL_MAIN) {
			if ((vol_main-=2)<0)
                           vol_main = 0;
			set_volume(VOL_LOCAL,(channel->volume*vol_main)/100);
			if(verbose)fprintf(stderr, "Main volume at %i.\r\n",vol_main);
		}
}

void increase_volume(int control)
{
	if (control==VOL_TUNER) {
		if ((tuner_vol+=300)>65535)
			tuner_vol = 65535;
		set_volume(VOL_TUNER,tuner_vol);
		if(verbose)fprintf(stderr, "Tuner volume at %i.\r\n",tuner_vol);
                return;
        };
	if (!audio_disabled)
		if (control==VOL_GLOBAL) {
			if ((vol_up+=2)>100)
				vol_up = 100;
			set_volume(VOL_GLOBAL,vol_up);
			if(verbose)fprintf(stderr, "Audio volume at %i.\r\n",vol_up);
		}
		if (control==VOL_LOCAL) {
			if ((channel->volume+=2)>100)
				channel->volume = 100;
			set_volume(VOL_LOCAL,channel->volume);
			if(verbose)fprintf(stderr, "Channel volume at %i.\r\n",channel->volume);
		}
		if (control==VOL_MAIN) {
			if ((vol_main+=2)>200)
				vol_main = 200;
			set_volume(VOL_LOCAL,(channel->volume*vol_main)/100);
			if(verbose)fprintf(stderr, "Main volume at %i.\r\n",vol_main);
		}
}

void switch_source(void)
{
	if (++cur_source >= vcap.channels)
		cur_source = 0;
	if(ioctl(fbttv, VIDIOCSCHAN,&vchan[cur_source]))
           	fail("VIDIOCSCHAN failed \n");
	if(verbose)fprintf(stderr, "Video Source: %s.\r\n",vchan[cur_source].name);
}

void process_channelkeys(struct channel_t **channel, int c)
{
   	if(c!=(*channel)->nr) {
        	*channel = find_channel(*channel, c);
                change_channel(*channel);
        };
}
 
int drawtext(int xpos,int ypos,int siz,char *str, int fcol, int bgcol, int bpp);

int tv_getchar(void)
{
   int c;
   int d[256];
   int i=0;
   
   switch(target) {
#ifdef GLIDE
   	case TARGET_GLIDE:
           c=(unsigned char)lin_getch();
           break;
#endif
#ifdef SVGALIB
        case TARGET_SVGALIB:
           c=getchar();
           break;
#endif
#ifdef FB
        case TARGET_FB:
           c=getchar();
           break;
#endif
        default:
           c=getchar();
           break;
   };
   if(c!=27)return c;

   switch(target) {
#ifdef GLIDE
   	case TARGET_GLIDE:
           d[0]=(unsigned char)lin_getch();
           break;
#endif
#ifdef SVGALIB
        case TARGET_SVGALIB:
#endif
#ifdef FB
        case TARGET_FB:
#endif
        default:
           d[0]=getchar();
           break;
   };
   
   for(;(d[i]!='~')&&((d[i]<'A')||(d[i]>'E'))&&(i<255);) {
       i++;
       switch(target) {
#ifdef GLIDE
   	    case TARGET_GLIDE:
               d[i]=(unsigned char)lin_getch();
               break;
#endif
#ifdef SVGALIB
            case TARGET_SVGALIB:
#endif
#ifdef FB
            case TARGET_FB:
#endif
            default:
               d[i]=getchar();
               break;
       };
   };
   if((d[0]=='[')&&(d[1]>='A')&&(d[1]<='D'))return K_UP+d[1]-'A';
   if((d[0]=='[')&&(d[1]=='[')&&(d[2]>='A')&&(d[2]<='E'))return K_F1+d[2]-'A';
   if((d[0]=='[')&&(d[2]=='~')&&(d[1]>='1')&&(d[1]<='6'))return K_HOME+d[1]-'1';
   if((d[0]=='[')&&(d[1]='1')&&(d[3]='~'))return K_F6+d[2]-'7';
   if((d[0]=='[')&&(d[1]='2')&&(d[3]='~'))return K_F9+d[2]-'0'-(d[2]>'2');

   return c;
};

void tv_drawscansegment(char *colors, int x ,int y, int bytes)
{
   if(x*bpp>linewidth)return;
   if(x*bpp+bytes>linewidth)bytes=linewidth-(x*bpp);
   switch(target) {
#ifdef FB
   	case TARGET_FB:
#endif
#ifdef GLIDE
   	case TARGET_GLIDE:
/*           grLfbWriteRegion(GR_BUFFER_FRONTBUFFER,x,y,GR_LFB_SRC_FMT_565,bytes/2,1,bytes,colors);*/
   	     memcpy(graphmem+linewidth*y+bpp*x,colors,bytes);	
           break;
#endif
#ifdef SVGALIB
        case TARGET_SVGALIB:
           vga_drawscansegment(colors,x,y,bytes);
           break;
#endif
   };
};

int main(int argc, char **argv)
{
  int c,d;
  struct channel_t *jump;
  char s[256], zero[4096], name[128]; 
  int i;
  int show_control=1, old_show_control;
  int edit_name=0;
  init(argc,argv);
  
  memset(zero,0,4096);
  jump = channel = channels;
  change_channel(channel);
  
  while (1)
	{
                for(i=576;i<600;i++)tv_drawscansegment(zero,0,i,linewidth);
                switch(show_control) {
                   case 1:
                      sprintf(s,"con:%4i\tbri:%4i\thue:%4i\tcol:%4i\tcvol:%3i\tmvol:%3i\tChn: %i",
	                    channel->contrast,
	                    channel->brightness,
	                    channel->hue,
	                    channel->colour,
                            channel->volume,
                            vol_main,
	                    channel->nr);
                      break;
                   case 2:
                      sprintf(s,"Freq:%s,%iKHz%+i\tChannel: %s",freq_to_str((channel->freq*125)>>1),
                         				     (channel->freq*125)>>1,channel->finetune,channel->name);
                      break;
                   case 3:
                      sprintf(s,"Master Vol:%i\t\tTuner Vol: %i",
                         				     vol_up,tuner_vol);
                      break;
                };
                if(show_control)drawtext(0,577,5,s,0xf800,0,bpp);
                d = tv_getchar();
                c=translate(d);

                if(edit_name){
                   if((d==10)||(d==13)||(c==59)){
                      free(channel->name);
                      edit_name=0;
                      show_control=old_show_control;
                      channel->name=strdup(name);
                      continue;
                   };
                   if(((d==127)||(d==8))&&(strlen(name)>0)){
                      name[strlen(name)-1]=0;
                      free(channel->name);
                      channel->name=strdup(name);
                      continue;
                   };
                   if((d>31)&&(d<256)&&(d!=127)&&(strlen(name)<126)){
                      name[strlen(name)+1]=0;
                      name[strlen(name)]=d;
                      free(channel->name);
                      channel->name=strdup(name);
                      continue;
                   };
                } else
                if(c<20)process_channelkeys(&channel,c+1); else 
		switch (c-20) {
                 case 13: 
			 increase_volume(VOL_MAIN);
			 break;
                 case 14: 
                         decrease_volume(VOL_MAIN);
			 break;
                 case 23: 
			 increase_volume(VOL_GLOBAL);
			 break;
                 case 24: 
                         decrease_volume(VOL_GLOBAL);
			 break;
                 case 25: 
			 increase_volume(VOL_TUNER);
			 break;
                 case 26: 
                         decrease_volume(VOL_TUNER);
			 break;
                 case 27:
                    	 channel->finetune++;
			 change_channel(channel);
                         break;
                 case 28:
                    	 channel->finetune--;
			 change_channel(channel);
                         break;
                 case 29:if(channel->finetune) {
                      	   channel->finetune=0;
			   change_channel(channel);
                         };  
                         break;
                 case 30:
                    	 i=next_freq(freq_to_num((channel->freq*125)/2));
                    	 channel->freq=(tvtuner[i].freq[cur_method]*2)/125;
			 change_channel(channel);
                         break;
                 case 31:
                    	 channel->freq=(tvtuner[prev_freq(freq_to_num((channel->freq*125)/2))].freq[cur_method]*2)/125;
			 change_channel(channel);
                         break;
                 case 32:{
                    	    struct channel_t *chn, *tmp;
                            chn = malloc(sizeof(struct channel_t));
                            memcpy(chn,channel,sizeof(struct channel_t));
                            tmp=channels->prev;
                            channels->prev=chn;
                            tmp->next=chn;
                            chn->prev=tmp;
                            chn->next=channels;
                            chn->nr=tmp->nr+1;
                            chn->name=malloc(strlen(channel->name)+4);
                            strcpy(chn->name,"CPY ");
                            strcat(chn->name,channel->name);
                        };
                    	break;
                 case 34:{
                    	FILE *f;
                        char fname[17];
                        struct stat st;
                        
                        i=0;
                        do {
                           sprintf(fname,"capture%03i.ppm",i++);
                        } while(((stat(fname,&st)==0)||(errno!=ENOENT))&&(i<1000));
                           
                        f=fopen(fname,"w+b");
                        
                        writeppm(f);
                        
                        fclose(f);
                        };
                    	break;
#ifdef JPEG
                 case 41:{
                    	FILE *f;
                        char fname[17];
                        struct stat st;
                        
                        i=0;
                        do {
                           sprintf(fname,"capture%03i.jpeg",i++);
                        } while(((stat(fname,&st)==0)||(errno!=ENOENT))&&(i<1000));
                           
                        f=fopen(fname,"w+b");
                        
                        writejpeg(f);
                        
                        fclose(f);
                        };
                    	break;
#endif
                 case 40:{
                    	FILE *f;
                        char fname[17];
                        struct stat st;
                        
                        i=0;
                        do {
                           sprintf(fname,"capture%03i.raw",i++);
                        } while(((stat(fname,&st)==0)||(errno!=ENOENT))&&(i<1000));
                           
                        f=fopen(fname,"w+b");
                        
                        writeraw(f);

                        fclose(f);
                        };
                    	break;
                 case 35:if(channel->next!=channel){
                            channel->prev->next=channel->next;
                            channel->next->prev=channel->prev;
                            channel->next=channels;
                            channel->prev=channels->prev;
                            channels->prev->next=channel;
                            channels->prev=channel;
                            channels=channel;
                            renumber_channels();
                        };
                    	break;
                 case 36:if(channel->next!=channel){
                            channel->prev->next=channel->next;
                            channel->next->prev=channel->prev;
                            channel->next=channels;
                            channel->prev=channels->prev;
                            channels->prev->next=channel;
                            channels->prev=channel;
                            renumber_channels();
                        };
                    	break;
                 case 37:if(channel->next!=channel){
                            struct channel_t *chn;

                            channel->prev->next=channel->next;
                            channel->next->prev=channel->prev;
                            chn=channel;
                            channel=channel->next;
                            free(chn);
                            renumber_channels();
                        };
                    	break;                 
                 case 38:if(channel->next!=channel){
                            struct channel_t *chn=channel->next->next;
                 
                            channel->prev->next=channel->next;
                            channel->next->next->prev=channel;
                            channel->next->next=channel;
                            channel->next->prev=channel->prev;
                            channel->prev=channel->next;
                            channel->next=chn;
                            if(channel==channels)channels=channel->prev;
                            renumber_channels();
                        };
                    	break;
                 case 39:
                        edit_name=1;
                        old_show_control=show_control;
                        show_control=2;
                        strcpy(name,channel->name);
                        break;
                 case 6: 
                      	 show_control=(show_control+1)%4;
                         break;
		 case 7:
			 channel = channel->next;
			 change_channel(channel);
			 break;
		 case 8:
			 channel = channel->prev;
			 change_channel(channel);
			 break;
		 case 9:
			 jump=channel;
			 break;
		 case 10:
			 channel=jump;
			 change_channel(channel);
			 break;
		 case 33:
			 toggle_capturing(1);
			 break;
		 case 0:
			 toggle_capturing(0);
			 break;
		 case 1:
			 toggle_audio();
			 break;
		 case 12:
			 decrease_volume(VOL_LOCAL);
			 break;
		 case 11:
			 increase_volume(VOL_LOCAL);
			 break;
		 case 2:
			 switch_source();
			 break;
		 case 3:
			 closedown();
			 break;
		 case 4:if((channel->contrast!=255)||(channel->brightness!=0)||
                             (channel->hue!=0)||(channel->colour!=255)) {
                    	     channel->contrast=255;
                    	     channel->brightness=0;
                    	     channel->hue=0;
                    	     channel->colour=255;
                             set_pict_params(channel);
                          };
                         break;
                 case 5:
                    	write_config();
                        break;
                 case 16:
                         if(channel->contrast>0) {
                    	    channel->contrast--;
                            set_pict_params(channel);
                         };
                         break;
                 case 15:
                         if(channel->contrast<511) {
                    	    channel->contrast++;
                            set_pict_params(channel);
                         };
                         break;
                 case 18:
                         if(channel->brightness>-127) {
                    	    channel->brightness--;
                            set_pict_params(channel);
                         };
                         break;
                 case 17:
                         if(channel->brightness<127) {
                    	    channel->brightness++;
                            set_pict_params(channel);
                         };
                         break;
                 case 20:
                         if(channel->hue>-127) {
                    	    channel->hue--;
                            set_pict_params(channel);
                         };
                         break;
                 case 19:
                         if(channel->hue<127) {
                    	    channel->hue++;
                            set_pict_params(channel);
                         };
                         break;
                 case 22:
                         if(channel->colour>0) {
                            channel->colour--;
                            set_pict_params(channel);
                         };           
                         break;
                 case 21:
                         if(channel->colour<511) {
                    	    channel->colour++;
                            set_pict_params(channel);
                         };
                         break;
		}
	}
	return 0;
}
