/* Table of opcodes for the OpenRISC 1000 family, OR32 ISA.
   Copyright 1990, 1991, 1992, 1993 Free Software Foundation, Inc.
   Contributed by Damjan Lampret (lampret@opencores.org).
   
This file is part of GDB and GAS.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  */

/* We treat all letters the same in encode/decode routines so
   we need to assign some characteristics to them like signess etc.*/

#define NUM_UNSIGNED 0
#define NUM_SIGNED 1

struct or32_letter {
        char letter;
        int  sign;
        /* int  reloc; relocation per letter ??*/
};

static struct or32_letter or32_letters[] =
{
{ 'A', NUM_UNSIGNED },
{ 'B', NUM_UNSIGNED },
{ 'D', NUM_UNSIGNED },
{ 'I', NUM_SIGNED },
{ 'K', NUM_UNSIGNED },
{ 'L', NUM_UNSIGNED },
{ 'N', NUM_SIGNED },
{ '\0', 0 }     /* dummy entry */
};

struct or32_opcode {
  /* Name of the instruction.  */
  char *name;

  /* A string of characters which describe the operands.
     Valid characters are:
     ,() Itself.  Characters appears in the assembly code.
     rA	 Register operand.
     rB  Register operand.
     rD  Register operand.
     I	 An immediate operand, range -32768 to 32767.
     J	 An immediate operand, range . (unused)
     K	 An immediate operand, range 0 to 65535.
     L	 An immediate operand, range 0 to 31.
     M	 An immediate operand, range . (unused)
     N	 An immediate operand, range -33554432 to 33554431.
     O	 An immediate operand, range . (unused) */
  char *args;
  
  /* Opcode and operand encoding. */
  char *encoding;
};

#ifndef CONST
#define CONST
#endif /* CONST */

/* Opcode encoding:
   machine[31:30]: first two bits of opcode
   		   00 - neither of source operands is GPR
   		   01 - second source operand is GPR (rB)
   		   10 - first source operand is GPR (rA)
   		   11 - both source operands are GPRs (rA and rB)
   machine[29:26]: next four bits of opcode
   machine[25:00]: instruction operands (specific to individual instruction)
*/

static CONST struct or32_opcode or32_opcodes[] =
{

{ "l.j",       "N",            "00 0x0  NNNNN NNNNN NNNN NNNN NNNN NNNN", },
{ "l.jal",     "N",            "00 0x1  NNNNN NNNNN NNNN NNNN NNNN NNNN", },
{ "l.bal",     "N",            "00 0x2  NNNNN NNNNN NNNN NNNN NNNN NNNN", },
{ "l.bnf",     "N",            "00 0x3  NNNNN NNNNN NNNN NNNN NNNN NNNN", },
{ "l.bf",      "N",            "00 0x4  NNNNN NNNNN NNNN NNNN NNNN NNNN", },
{ "l.nop",     "",             "00 0x5  01--- ----- ---- ---- ---- ----", },
{ "l.sys",     "K",            "00 0x5  10--- ----- KKKK KKKK KKKK KKKK", },
{ "l.brk",     "K",            "00 0x5  11--- ----- KKKK KKKK KKKK KKKK", },
{ "l.movhi",   "rD,K",         "00 0x6  DDDDD ----- KKKK KKKK KKKK KKKK", },
{ "l.mfsr",    "rD,rA",        "00 0x7  DDDDD AAAAA ---- ---- ---- ----", },

{ "lv.all_eq.b","rD,rA,rB",    "00 0xA  DDDDD AAAAA BBBB B--- 0x1 0x0", },
{ "lv.all_eq.h","rD,rA,rB",    "00 0xA  DDDDD AAAAA BBBB B--- 0x1 0x1", },
{ "lv.all_ge.b","rD,rA,rB",    "00 0xA  DDDDD AAAAA BBBB B--- 0x1 0x2", },
{ "lv.all_ge.h","rD,rA,rB",    "00 0xA  DDDDD AAAAA BBBB B--- 0x1 0x3", },
{ "lv.all_gt.b","rD,rA,rB",    "00 0xA  DDDDD AAAAA BBBB B--- 0x1 0x4", },
{ "lv.all_gt.h","rD,rA,rB",    "00 0xA  DDDDD AAAAA BBBB B--- 0x1 0x5", },
{ "lv.all_le.b","rD,rA,rB",    "00 0xA  DDDDD AAAAA BBBB B--- 0x1 0x6", },
{ "lv.all_le.h","rD,rA,rB",    "00 0xA  DDDDD AAAAA BBBB B--- 0x1 0x7", },
{ "lv.all_lt.b","rD,rA,rB",    "00 0xA  DDDDD AAAAA BBBB B--- 0x1 0x8", },
{ "lv.all_lt.h","rD,rA,rB",    "00 0xA  DDDDD AAAAA BBBB B--- 0x1 0x9", },
{ "lv.all_ne.b","rD,rA,rB",    "00 0xA  DDDDD AAAAA BBBB B--- 0x1 0xA", },
{ "lv.all_ne.h","rD,rA,rB",    "00 0xA  DDDDD AAAAA BBBB B--- 0x1 0xB", },
{ "lv.any_eq.b","rD,rA,rB",    "00 0xA  DDDDD AAAAA BBBB B--- 0x2 0x0", },
{ "lv.any_eq.h","rD,rA,rB",    "00 0xA  DDDDD AAAAA BBBB B--- 0x2 0x1", },
{ "lv.any_ge.b","rD,rA,rB",    "00 0xA  DDDDD AAAAA BBBB B--- 0x2 0x2", },
{ "lv.any_ge.h","rD,rA,rB",    "00 0xA  DDDDD AAAAA BBBB B--- 0x2 0x3", },
{ "lv.any_gt.b","rD,rA,rB",    "00 0xA  DDDDD AAAAA BBBB B--- 0x2 0x4", },
{ "lv.any_gt.h","rD,rA,rB",    "00 0xA  DDDDD AAAAA BBBB B--- 0x2 0x5", },
{ "lv.any_le.b","rD,rA,rB",    "00 0xA  DDDDD AAAAA BBBB B--- 0x2 0x6", },
{ "lv.any_le.h","rD,rA,rB",    "00 0xA  DDDDD AAAAA BBBB B--- 0x2 0x7", },
{ "lv.any_lt.b","rD,rA,rB",    "00 0xA  DDDDD AAAAA BBBB B--- 0x2 0x8", },
{ "lv.any_lt.h","rD,rA,rB",    "00 0xA  DDDDD AAAAA BBBB B--- 0x2 0x9", },
{ "lv.any_ne.b","rD,rA,rB",    "00 0xA  DDDDD AAAAA BBBB B--- 0x2 0xA", },
{ "lv.any_ne.h","rD,rA,rB",    "00 0xA  DDDDD AAAAA BBBB B--- 0x2 0xB", },
{ "lv.add.b",  "rD,rA,rB",     "00 0xA  DDDDD AAAAA BBBB B--- 0x3 0x0", },
{ "lv.add.h",  "rD,rA,rB",     "00 0xA  DDDDD AAAAA BBBB B--- 0x3 0x1", },
{ "lv.adds.b", "rD,rA,rB",     "00 0xA  DDDDD AAAAA BBBB B--- 0x3 0x2", },
{ "lv.adds.h", "rD,rA,rB",     "00 0xA  DDDDD AAAAA BBBB B--- 0x3 0x3", },
{ "lv.and",    "rD,rA,rB",     "00 0xA  DDDDD AAAAA BBBB B--- 0x3 0x4", },
{ "lv.avg.b",  "rD,rA,rB",     "00 0xA  DDDDD AAAAA BBBB B--- 0x3 0x5", },
{ "lv.avg.h",  "rD,rA,rB",     "00 0xA  DDDDD AAAAA BBBB B--- 0x3 0x6", },
{ "lv.cmpeq.b","rD,rA,rB",     "00 0xA  DDDDD AAAAA BBBB B--- 0x4 0x0", },
{ "lv.cmpeq.h","rD,rA,rB",     "00 0xA  DDDDD AAAAA BBBB B--- 0x4 0x1", },
{ "lv.cmpge.b","rD,rA,rB",     "00 0xA  DDDDD AAAAA BBBB B--- 0x4 0x2", },
{ "lv.cmpge.h","rD,rA,rB",     "00 0xA  DDDDD AAAAA BBBB B--- 0x4 0x3", },
{ "lv.cmpgt.b","rD,rA,rB",     "00 0xA  DDDDD AAAAA BBBB B--- 0x4 0x4", },
{ "lv.cmpgt.h","rD,rA,rB",     "00 0xA  DDDDD AAAAA BBBB B--- 0x4 0x5", },
{ "lv.cmple.b","rD,rA,rB",     "00 0xA  DDDDD AAAAA BBBB B--- 0x4 0x6", },
{ "lv.cmple.h","rD,rA,rB",     "00 0xA  DDDDD AAAAA BBBB B--- 0x4 0x7", },
{ "lv.cmplt.b","rD,rA,rB",     "00 0xA  DDDDD AAAAA BBBB B--- 0x4 0x8", },
{ "lv.cmplt.h","rD,rA,rB",     "00 0xA  DDDDD AAAAA BBBB B--- 0x4 0x9", },
{ "lv.cmpne.b","rD,rA,rB",     "00 0xA  DDDDD AAAAA BBBB B--- 0x4 0xA", },
{ "lv.cmpne.h","rD,rA,rB",     "00 0xA  DDDDD AAAAA BBBB B--- 0x4 0xB", },
{ "lv.itof",   "rD,rA,rB",     "00 0xA  DDDDD AAAAA BBBB B--- 0x5 0x0", },
{ "lv.ftoi",   "rD,rA,rB",     "00 0xA  DDDDD AAAAA BBBB B--- 0x5 0x1", },
{ "lv.madd.h", "rD,rA,rB",     "00 0xA  DDDDD AAAAA BBBB B--- 0x5 0x2", },
{ "lv.max.b",  "rD,rA,rB",     "00 0xA  DDDDD AAAAA BBBB B--- 0x5 0x3", },
{ "lv.max.h",  "rD,rA,rB",     "00 0xA  DDDDD AAAAA BBBB B--- 0x5 0x4", },
{ "lv.merge.b","rD,rA,rB",     "00 0xA  DDDDD AAAAA BBBB B--- 0x5 0x5", },
{ "lv.merge.h","rD,rA,rB",     "00 0xA  DDDDD AAAAA BBBB B--- 0x5 0x6", },
{ "lv.min.b",  "rD,rA,rB",     "00 0xA  DDDDD AAAAA BBBB B--- 0x5 0x7", },
{ "lv.min.h",  "rD,rA,rB",     "00 0xA  DDDDD AAAAA BBBB B--- 0x5 0x8", },
{ "lv.msub.h", "rD,rA,rB",     "00 0xA  DDDDD AAAAA BBBB B--- 0x5 0x9", },
{ "lv.nand.b", "rD,rA,rB",     "00 0xA  DDDDD AAAAA BBBB B--- 0x5 0xA", },
{ "lv.nand.h", "rD,rA,rB",     "00 0xA  DDDDD AAAAA BBBB B--- 0x5 0xB", },
{ "lv.nor.b",  "rD,rA,rB",     "00 0xA  DDDDD AAAAA BBBB B--- 0x5 0xC", },
{ "lv.nor.h",  "rD,rA,rB",     "00 0xA  DDDDD AAAAA BBBB B--- 0x5 0xD", },
{ "lv.or.b",   "rD,rA,rB",     "00 0xA  DDDDD AAAAA BBBB B--- 0x5 0xE", },
{ "lv.or.h",   "rD,rA,rB",     "00 0xA  DDDDD AAAAA BBBB B--- 0x5 0xF", },
{ "lv.pack.b", "rD,rA,rB",     "00 0xA  DDDDD AAAAA BBBB B--- 0x5 0x0", },
{ "lv.pack.h", "rD,rA,rB",     "00 0xA  DDDDD AAAAA BBBB B--- 0x6 0x1", },
{ "lv.packs.b","rD,rA,rB",     "00 0xA  DDDDD AAAAA BBBB B--- 0x6 0x2", },
{ "lv.packs.h","rD,rA,rB",     "00 0xA  DDDDD AAAAA BBBB B--- 0x6 0x3", },
{ "lv.perm.b", "rD,rA,rB",     "00 0xA  DDDDD AAAAA BBBB B--- 0x6 0x4", },
{ "lv.perm.h", "rD,rA,rB",     "00 0xA  DDDDD AAAAA BBBB B--- 0x6 0x5", },
{ "lv.rl.b",   "rD,rA,rB",     "00 0xA  DDDDD AAAAA BBBB B--- 0x6 0x6", },
{ "lv.rl.h",   "rD,rA,rB",     "00 0xA  DDDDD AAAAA BBBB B--- 0x6 0x7", },
{ "lv.sl.b",   "rD,rA,rB",     "00 0xA  DDDDD AAAAA BBBB B--- 0x6 0x8", },
{ "lv.sl.h",   "rD,rA,rB",     "00 0xA  DDDDD AAAAA BBBB B--- 0x6 0x9", },
{ "lv.sle.b",  "rD,rA,rB",     "00 0xA  DDDDD AAAAA BBBB B--- 0x6 0xA", },
{ "lv.sle.h",  "rD,rA,rB",     "00 0xA  DDDDD AAAAA BBBB B--- 0x6 0xB", },
{ "lv.splat.b","rD,rA,rB",     "00 0xA  DDDDD AAAAA BBBB B--- 0x6 0xC", },
{ "lv.splat.h","rD,rA,rB",     "00 0xA  DDDDD AAAAA BBBB B--- 0x6 0xD", },
{ "lv.sr.b",   "rD,rA,rB",     "00 0xA  DDDDD AAAAA BBBB B--- 0x6 0xE", },
{ "lv.sr.h",   "rD,rA,rB",     "00 0xA  DDDDD AAAAA BBBB B--- 0x6 0xF", },
{ "lv.sra.b",  "rD,rA,rB",     "00 0xA  DDDDD AAAAA BBBB B--- 0x6 0x0", },
{ "lv.sra.h",  "rD,rA,rB",     "00 0xA  DDDDD AAAAA BBBB B--- 0x6 0x1", },
{ "lv.sre.b",  "rD,rA,rB",     "00 0xA  DDDDD AAAAA BBBB B--- 0x6 0x2", },
{ "lv.sre.h",  "rD,rA,rB",     "00 0xA  DDDDD AAAAA BBBB B--- 0x6 0x3", },
{ "lv.sub.b",  "rD,rA,rB",     "00 0xA  DDDDD AAAAA BBBB B--- 0x6 0x4", },
{ "lv.sub.h",  "rD,rA,rB",     "00 0xA  DDDDD AAAAA BBBB B--- 0x6 0x5", },
{ "lv.subs.b", "rD,rA,rB",     "00 0xA  DDDDD AAAAA BBBB B--- 0x6 0x6", },
{ "lv.subs.h", "rD,rA,rB",     "00 0xA  DDDDD AAAAA BBBB B--- 0x6 0x7", },
{ "lv.unpack.b","rD,rA,rB",    "00 0xA  DDDDD AAAAA BBBB B--- 0x6 0x8", },
{ "lv.unpack.h","rD,rA,rB",    "00 0xA  DDDDD AAAAA BBBB B--- 0x6 0x9", },
{ "lv.xor.b",  "rD,rA,rB",     "00 0xA  DDDDD AAAAA BBBB B--- 0x6 0xA", },
{ "lv.xor.h",  "rD,rA,rB",     "00 0xA  DDDDD AAAAA BBBB B--- 0x6 0xB", },

{ "lf.ftoi.s",  "rD,rA",       "00 0xB  DDDDD AAAAA BBBB B--- 0x1 0x0", },
{ "lf.add.s",   "rD,rA,rB",    "00 0xB  DDDDD AAAAA BBBB B--- 0x1 0x1", },
{ "lf.sub.s",   "rD,rA,rB",    "00 0xB  DDDDD AAAAA BBBB B--- 0x1 0x2", },
{ "lf.mul.s",   "rD,rA,rB",    "00 0xB  DDDDD AAAAA BBBB B--- 0x1 0x3", },
{ "lf.div.s",   "rD,rA,rB",    "00 0xB  DDDDD AAAAA BBBB B--- 0x1 0x4", },
{ "lf.madd.s",  "rD,rA,rB",    "00 0xB  DDDDD AAAAA BBBB B--- 0x1 0x5", },
{ "lf.sfeq.s",  "rA,rB",       "00 0xB  ----- AAAAA BBBB B--- 0x1 0x6", },
{ "lf.sfne.s",  "rA,rB",       "00 0xB  ----- AAAAA BBBB B--- 0x1 0x7", },
{ "lf.sfgt.s",  "rA,rB",       "00 0xB  ----- AAAAA BBBB B--- 0x1 0x8", },
{ "lf.sfge.s",  "rA,rB",       "00 0xB  ----- AAAAA BBBB B--- 0x1 0x9", },
{ "lf.sflt.s",  "rA,rB",       "00 0xB  ----- AAAAA BBBB B--- 0x1 0xA", },
{ "lf.sfle.s",  "rA,rB",       "00 0xB  ----- AAAAA BBBB B--- 0x1 0xB", },

{ "l.mtsr",    "rA,rB",        "01 0x0  ----- AAAAA BBBB B--- ---- ----", },
{ "l.jr",      "rB",           "01 0x1  ----- ----- BBBB B--- ---- ----", },
{ "l.jalr",    "rB",           "01 0x2  ----- ----- BBBB B--- ---- ----", },
{ "l.rfe",     "rB",           "01 0x3  ----- ----- BBBB B--- ---- ----", },
{ "lf.itof.s", "rD,rA",        "01 0x4  DDDDD ----- BBBB B--- ---- ----", },

{ "l.lw",      "rD,I(rA)",     "10 0x0  DDDDD AAAAA IIII IIII IIII IIII", },
{ "l.lbz",     "rD,I(rA)",     "10 0x1  DDDDD AAAAA IIII IIII IIII IIII", },
{ "l.lbs",     "rD,I(rA)",     "10 0x2  DDDDD AAAAA IIII IIII IIII IIII", },
{ "l.lhz",     "rD,I(rA)",     "10 0x3  DDDDD AAAAA IIII IIII IIII IIII", },
{ "l.lhs",     "rD,I(rA)",     "10 0x4  DDDDD AAAAA IIII IIII IIII IIII", },

{ "l.addi",    "rD,rA,I",      "10 0x5  DDDDD AAAAA IIII IIII IIII IIII", },
{ "l.addic",   "rD,rA,I",      "10 0x6  DDDDD AAAAA IIII IIII IIII IIII", },
{ "l.subi",    "rD,rA,I",      "10 0x7  DDDDD AAAAA IIII IIII IIII IIII", },
{ "l.andi",    "rD,rA,K",      "10 0x8  DDDDD AAAAA KKKK KKKK KKKK KKKK", },
{ "l.ori",     "rD,rA,K",      "10 0x9  DDDDD AAAAA KKKK KKKK KKKK KKKK", },
{ "l.xori",    "rD,rA,I",      "10 0xA  DDDDD AAAAA IIII IIII IIII IIII", },
{ "l.muli",    "rD,rA,I",      "10 0xB  DDDDD AAAAA IIII IIII IIII IIII", },
{ "l.maci",    "rD,rA,I",      "10 0xC  DDDDD AAAAA IIII IIII IIII IIII", },
{ "l.slli",    "rD,rA,L",      "10 0xD  DDDDD AAAAA ---- ---- 000L LLLL", },
{ "l.srli",    "rD,rA,L",      "10 0xD  DDDDD AAAAA ---- ---- 001L LLLL", },
{ "l.srai",    "rD,rA,L",      "10 0xD  DDDDD AAAAA ---- ---- 010L LLLL", },
{ "l.rori",    "rD,rA,L",      "10 0xD  DDDDD AAAAA ---- ---- 100L LLLL", },

{ "l.sfeqi",   "rA,I",         "10 0xE  00000 AAAAA IIII IIII IIII IIII", },
{ "l.sfnei",   "rA,I",         "10 0xE  00001 AAAAA IIII IIII IIII IIII", },
{ "l.sfgtui",  "rA,I",         "10 0xE  00010 AAAAA IIII IIII IIII IIII", },
{ "l.sfgeui",  "rA,I",         "10 0xE  00011 AAAAA IIII IIII IIII IIII", },
{ "l.sfltui",  "rA,I",         "10 0xE  00100 AAAAA IIII IIII IIII IIII", },
{ "l.sfleui",  "rA,I",         "10 0xE  00101 AAAAA IIII IIII IIII IIII", },
{ "l.sfgtsi",  "rA,I",         "10 0xE  01010 AAAAA IIII IIII IIII IIII", },
{ "l.sfgesi",  "rA,I",         "10 0xE  01011 AAAAA IIII IIII IIII IIII", },
{ "l.sfltsi",  "rA,I",         "10 0xE  01100 AAAAA IIII IIII IIII IIII", },
{ "l.sflesi",  "rA,I",         "10 0xE  01101 AAAAA IIII IIII IIII IIII", },

{ "l.sw",      "I(rA),rB",     "11 0x5  IIIII AAAAA BBBB BIII IIII IIII", },
{ "l.sb",      "I(rA),rB",     "11 0x6  IIIII AAAAA BBBB BIII IIII IIII", },
{ "l.sh",      "I(rA),rB",     "11 0x7  IIIII AAAAA BBBB BIII IIII IIII", },

{ "l.add",     "rD,rA,rB",     "11 0x8  DDDDD AAAAA BBBB B--- ---- 0x0", },
{ "l.addc",    "rD,rA,rB",     "11 0x8  DDDDD AAAAA BBBB B--- ---- 0x1", },
{ "l.sub",     "rD,rA,rB",     "11 0x8  DDDDD AAAAA BBBB B--- ---- 0x2", },
{ "l.and",     "rD,rA,rB",     "11 0x8  DDDDD AAAAA BBBB B--- ---- 0x3", },
{ "l.or",      "rD,rA,rB",     "11 0x8  DDDDD AAAAA BBBB B--- ---- 0x4", },
{ "l.xor",     "rD,rA,rB",     "11 0x8  DDDDD AAAAA BBBB B--- ---- 0x5", },
{ "l.mul",     "rD,rA,rB",     "11 0x8  DDDDD AAAAA BBBB B--- ---- 0x6", },
{ "l.mac",     "rD,rA,rB",     "11 0x8  DDDDD AAAAA BBBB B--- ---- 0x7", },
{ "l.sll",     "rD,rA,rB",     "11 0x8  DDDDD AAAAA BBBB B--- 000- 0x8", },
{ "l.srl",     "rD,rA,rB",     "11 0x8  DDDDD AAAAA BBBB B--- 001- 0x8", },
{ "l.sra",     "rD,rA,rB",     "11 0x8  DDDDD AAAAA BBBB B--- 010- 0x8", },
{ "l.ror",     "rD,rA,rB",     "11 0x8  DDDDD AAAAA BBBB B--- 100- 0x8", },
{ "l.div",     "rD,rA,rB",     "11 0x8  DDDDD AAAAA BBBB B--- ---- 0x9", },
{ "l.divu",    "rD,rA,rB",     "11 0x8  DDDDD AAAAA BBBB B--- ---- 0xA", },

{ "l.sfeq",    "rA,rB",        "11 0x9  00000 AAAAA BBBB B--- ---- ----", },
{ "l.sfne",    "rA,rB",        "11 0x9  00001 AAAAA BBBB B--- ---- ----", },
{ "l.sfgtu",   "rA,rB",        "11 0x9  00010 AAAAA BBBB B--- ---- ----", },
{ "l.sfgeu",   "rA,rB",        "11 0x9  00011 AAAAA BBBB B--- ---- ----", },
{ "l.sfltu",   "rA,rB",        "11 0x9  00100 AAAAA BBBB B--- ---- ----", },
{ "l.sfleu",   "rA,rB",        "11 0x9  00101 AAAAA BBBB B--- ---- ----", },
{ "l.sfgts",   "rA,rB",        "11 0x9  01010 AAAAA BBBB B--- ---- ----", },
{ "l.sfges",   "rA,rB",        "11 0x9  01011 AAAAA BBBB B--- ---- ----", },
{ "l.sflts",   "rA,rB",        "11 0x9  01100 AAAAA BBBB B--- ---- ----", },
{ "l.sfles",   "rA,rB",        "11 0x9  01101 AAAAA BBBB B--- ---- ----", },


{ "", "", "" }		/* Dummy entry, not included in NUM_OPCODES.  This
			   lets code examine entry i+1 without checking
			   if we've run off the end of the table.  */
};

CONST unsigned int num_opcodes = (((sizeof or32_opcodes) / (sizeof or32_opcodes[0])) - 1);

/* Calculates instruction length in bytes. Either 2 or 4 for OR16
   and always 4 for OR32. */
int insn_len(char *insn)
{
        struct or32_opcode *pinsn;
        char *enc;
        int len = 0;

        for(pinsn = or32_opcodes; strlen(pinsn->name); pinsn++) {
                if (strcmp(pinsn->name, insn) == 0) {
                        for (enc = pinsn->encoding; *enc != '\0'; enc++)
                                if ((*enc == '0') && (*(enc+1) == 'x')) {
                                        len += 4;
                                        enc += 2;
                                }
                                else if (!isspace(*enc))
                                        len++;
                        return len / 8;
                }
        }
        printf("insn_len(%s): Unknown instruction.\n", insn);
        exit(1);
}

/* Is individual insn's operand signed or unsigned? */
int letter_signed(char l)
{
        struct or32_letter *pletter;

        for(pletter = or32_letters; pletter->letter != '\0'; pletter++)
                if (pletter->letter == l)
                        return pletter->sign;

        printf("letter_signed(%c): Unknown letter.\n", l);
        return 0;
}

/* Number of letters in the individual lettered operand. */
int letter_range(char l)
{
        struct or32_opcode *pinsn;
        char *enc;
        int range = 0;

        for(pinsn = or32_opcodes; strlen(pinsn->name); pinsn++) {
                if (strchr(pinsn->encoding,l)) {
                        for (enc = pinsn->encoding; *enc != '\0'; enc++)
                                if ((*enc == '0') && (*(enc+1) == 'x')) {
                                        enc += 2;
                                }
                                else if (*enc == l)
                                        range++;
                        return range;
                }
        }
        printf("\nABORT: letter_range(%c): Never used letter.\n", l);
        exit(1);
}
