#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <getopt.h>
#include <byteswap.h>

#define BSWAP

/* big endian memory */

unsigned char ram[16*1024*1024];

#define PHYSMEM 0x40000000
#define PROGSTART 0x10000
#define IO 0xc0000000

#ifdef BSWAP

#define _MEM(x) (*(unsigned int *)(ram+x))
#define _MEM8(x) (ram[(x&0xfffffffc)+3-(x&3)])
#define _MEM16(x) (*(unsigned short *)(ram+(x&0xfffffffc)+2-(x&2)))

#define _SMEM(x,d) *(unsigned int *)(ram+x)=d
#define _SMEM8(x,d) ram[(x&0xfffffffc)+3-(x&3)]=d
#define _SMEM16(x,d) *(unsigned short *)(ram+(x&0xfffffffc)+2-(x&2))=d

#else /* BSWAP */

#define _MEM(x) ((ram[x]<<24) | (ram[x+1]<<16) | (ram[x+2]<<8) | ram[x+3])
#define _MEM8(x) (ram[x])
#define _MEM16(x) ((ram[x]<<8) | ram[x+1])

#define _SMEM8(x,d) ram[x]=d
#define _SMEM16(x,d) ram[x]=(d)>>8; ram[x+1]=(d)&0xff 
#define _SMEM(x,d) ram[x]=(d)>>24; ram[x+1]=((d)>>16)&0xff; ram[x+2]=((d)>>8)&0xff; ram[x+3]=(d)&0xff 

#endif /* BSWAP */

#define IMMU(x) (x)
unsigned int DMMU(unsigned int x) {
    if((x&0xfff00000)==0) {
        x+=65536;
    }
    return x;
}

unsigned int IMEM(unsigned int x) {
    x=IMMU(x);
    if(x<PHYSMEM) return _MEM(x);
}

unsigned int MEM(unsigned int x) {
    x=DMMU(x);
    if(x<PHYSMEM) return _MEM(x);
}

unsigned int MEM8(unsigned int x) {
    x=DMMU(x);
    if(x<PHYSMEM) return _MEM8(x);
    if(x==IO) return 0; /* no input for a while */
}

unsigned int MEM16(unsigned int x) {
    x=DMMU(x);
    if(x<PHYSMEM) return _MEM16(x);
}

void SMEM(unsigned int x, unsigned int d) {
    x=DMMU(x);
    if(x<PHYSMEM) {
        _SMEM(x,d); 
    } else {
    }
}

void SMEM8(unsigned int x, unsigned int d) {
    x=DMMU(x);
    if(x<PHYSMEM) {
        _SMEM8(x,d); 
    } else {
        if(x==IO) putchar(d);
    }
}

void SMEM16(unsigned int x, unsigned int d) {
    x=DMMU(x);
    if(x<PHYSMEM) {
        _SMEM16(x,d); 
    } else {
    }
}

/* sign extending */
#define SEX8(x) (x | ((x&0x80)?0xffffff00:0))
#define SEX16(x) (x | ((x&0x8000)?0xffff0000:0))
#define SEX26(x) (x | ((x&0x2000000)?0xfc000000:0))
#define SEX28(x) (x | ((x&0x8000000)?0xf0000000:0))

/* bit fields in instruction words */
#define OP(ins) ((ins>>26) & 0x3f)
#define N(ins) SEX26(ins&0x3ffffff)
#define I(ins) SEX16(ins&0xffff)
#define SI(ins) SEX16((ins&0x7ff)+((ins&0x3e00000)>>10))

#define K(ins) (ins&0xffff)
#define L(ins) (ins&0x1f)
#define B(ins) ((ins>>11)&0x1f)
#define A(ins) ((ins>>16)&0x1f)
#define D(ins) ((ins>>21)&0x1f)

/* link register is a special register */
#define _lr gpr[11]

#define POW(i) (1<<i)
#define POW1(i) ((1<<i)-1)
#define POW2(i) (~((1<<(32-i))-1))

unsigned int gpr[16];
unsigned int spr[32][2048];
unsigned int pc, flags, div0, carry;
long long mac;
unsigned int inscounter;

int usage(char *s) {
    fprintf(stderr, "%s\n",s);
    return 0;
}

int main(int argc, char *argv[]) {
    int c, f, i, j; 
    int nextjump;
    unsigned int newpc;
    int verbose;

    verbose=0;
    for (;;) {
        if (-1 == (c = getopt(argc, argv, "hvw")))
            break;
        switch (c) {
        case 'v':
            verbose=1;
            break;
        case 'w':
            verbose=2;
            break;
        case 'h':
        default:
            usage(argv[0]);
            exit(1);
        }
    }

    if (optind == argc) {
        usage(argv[0]);
        exit(1);
    }

    f=open("rom.or32", O_RDONLY);
    i=read(f, &ram[0], 64*1024);
#ifdef BSWAP
    if(i>0) {
        for(j=0;j<i;j+=4) {
            unsigned int k;
            k=*(unsigned int *)(ram+j);
            ram[j]=k>>24;
            ram[j+1]=(k>>16)&0xff;
            ram[j+2]=(k>>8)&0xff;
            ram[j+3]=k&0xff;
        }
    }
#endif
    close(f);

    f=open(argv[optind], O_RDONLY);
    i=read(f, &ram[PROGSTART], 1024*1024);
#ifdef BSWAP
    if(i>0) {
        for(j=0;j<i;j+=4) {
            unsigned int k;
            k=*(unsigned int *)(ram+PROGSTART+j);
            ram[PROGSTART+j]=k>>24;
            ram[PROGSTART+j+1]=(k>>16)&0xff;
            ram[PROGSTART+j+2]=(k>>8)&0xff;
            ram[PROGSTART+j+3]=k&0xff;
        }
    }
#endif
    close(f);
    
    inscounter=0;
    nextjump=0;
    pc=0x100;
    gpr[0]=0;

    while(1) {
        unsigned int ins, op;
        
        ins=IMEM(pc);
        inscounter++;
        if(verbose&1)printf("%8x: %08x   ", pc, ins);
        if(verbose&2)printf("%8x: %08x\n", pc, ins);
        op=OP(ins);
        switch(op) {
            case 0x00: /* l.j */
                if(verbose&1)printf("l.j      %7i\n",N(ins));
                newpc=pc + (N(ins)<<2);
                nextjump=-2;
                break;
            case 0x01: /* l.jal */
                if(verbose&1)printf("l.jal    %7i\n",N(ins));
                _lr=pc+8;
                newpc= pc + (N(ins)<<2);
                nextjump=-2;
                break;
            case 0x03: /* l.bnf */
                if(verbose&1)printf("l.bnf    %7i              f=%i\n",N(ins),flags&1);
                if(!(flags&1)) {
                    nextjump=-2;
                    newpc= pc + (N(ins)<<2);
                }
                break;
            case 0x04: /* l.bf */
                if(verbose&1)printf("l.bf     %7i              f=%i\n",N(ins),flags&1);
                if(flags&1) { 
                    nextjump=-2;
                    newpc= pc + (N(ins)<<2);
                }
                break;
            case 0x05: /* l.nop */
                if(verbose&1)printf("l.nop\n");
                /* XXX only if top 8 bits is 0x15 */
                break;

            case 0x06: /* l.movhi */
                if(verbose&1)printf("l.movhi  r%-2i,%4x\n",D(ins),K(ins));
                gpr[D(ins)] = K(ins)<<16;
                break;

            case 0x07: /* l.mfsr */
                if(verbose&1)printf("l.mfspr  %i(r%-2i),r%-2i     r%-2i=%08x mem=%08x\n",I(ins),A(ins),
                    		D(ins),A(ins),gpr[A(ins)],MEM(gpr[A(ins)]+I(ins)));
                /* XXX not done yet */
                break;
            case 0x08: /* l.sys */
                /* XXX only if top 20 bits is 0x20000 */
                /* sys K(ins) */
                if(verbose&1)printf("l.sys    %4x\n",K(ins));
                switch(K(ins)) {
                    case 203:
                        return 0;
                        break;
                    case 201:
                        gpr[3]=inscounter;
                        break;
                    default:
                        printf("unsupported l.sys %i\n",K(ins));
                }
                break;
            case 0x09: /* l.rfe */
                /* XXX not done yet */
                break;

            case 0x10: /* l.mtsr */
                if(verbose&1)printf("l.mtspr  %7i(r%-2i),r%-2i     r%-2i=%08x mem=%08x\n",I(ins),A(ins),
                    		D(ins),A(ins),gpr[A(ins)],MEM(gpr[A(ins)]+I(ins)));
                /* XXX not done yet */
                break;
            case 0x11: /* l.jr */
                if(verbose&1)printf("l.jr     r%-2i                  r%-2i=%08x\n",B(ins),B(ins),gpr[B(ins)]);
                nextjump=-2;
                newpc=gpr[B(ins)];
                break;
            case 0x12: /* l.jalr */
                if(verbose&1)printf("l.jlar   r%-2i                  r%-2i=%08x\n",B(ins),B(ins),gpr[B(ins)]);
                nextjump=-2;
                newpc=gpr[B(ins)];
                _lr=pc+8;
                break;
                
            case 0x20: /* l.lw */
                if(verbose&1)printf("l.lw     %7i(r%-2i),r%-2i     r%-2i=%08x mem=%08x\n",I(ins),A(ins),
                    		D(ins),A(ins),gpr[A(ins)],MEM(gpr[A(ins)]+I(ins)));
                gpr[D(ins)]=MEM(gpr[A(ins)]+I(ins));
                break;
            case 0x21: /* l.lbz */
                if(verbose&1)printf("l.lbz    %7i(r%-2i),r%-2i     r%-2i=%08x mem=%02x\n",I(ins),A(ins),
                    		D(ins),A(ins),gpr[A(ins)],MEM8(gpr[A(ins)]+I(ins)));
                gpr[D(ins)]=MEM8(gpr[A(ins)]+I(ins));
                break;
            case 0x22: /* l.lbs */
                if(verbose&1)printf("l.lbs    %7i(r%-2i),r%-2i     r%-2i=%08x mem=%02x\n",I(ins),A(ins),
                    		D(ins),A(ins),gpr[A(ins)],MEM8(gpr[A(ins)]+I(ins)));
                gpr[D(ins)]=SEX8(MEM8(gpr[A(ins)]+I(ins)));
                break;
            case 0x23: /* l.lhz */
                if(verbose&1)printf("l.lhz    %7i(r%-2i),r%-2i     r%-2i=%08x mem=%04x\n",I(ins),A(ins),
                    		D(ins),A(ins),gpr[A(ins)],MEM16(gpr[A(ins)]+I(ins)));
                gpr[D(ins)]=MEM16(gpr[A(ins)]+I(ins));
                break;
            case 0x24: /* l.lhs */
                if(verbose&1)printf("l.lhs    %7i(r%-2i),r%-2i     r%-2i=%08x mem=%04x\n",I(ins),A(ins),
                    		D(ins),A(ins),gpr[A(ins)],MEM16(gpr[A(ins)]+I(ins)));
                gpr[D(ins)]=SEX16(MEM16(gpr[A(ins)]+I(ins)));
                break;
                
            case 0x25: /* l.addi */
                if(verbose&1)printf("l.addi   r%-2i, r%-2i, %7i    r%-2i=%08x r%-2i=%08x\n",D(ins),A(ins),
                    		I(ins),D(ins),gpr[D(ins)],A(ins),gpr[A(ins)]);
                gpr[D(ins)]=gpr[A(ins)]+I(ins);
                break;
            case 0x26: /* l.addic */
                if(verbose&1)printf("l.addic  r%-2i, r%-2i, %7i    r%-2i=%08x r%-2i=%08x carry=%i\n",D(ins),A(ins),
                    		I(ins),D(ins),gpr[D(ins)],A(ins),gpr[A(ins)], carry);
                gpr[D(ins)]=gpr[A(ins)]+I(ins) + carry;
                break;
            case 0x27: /* l.subi */
                if(verbose&1)printf("l.subi   r%-2i, r%-2i, %7i    r%-2i=%08x r%-2i=%08x\n",D(ins),A(ins),
                    		I(ins),D(ins),gpr[D(ins)],A(ins),gpr[A(ins)]);
                gpr[D(ins)]=gpr[A(ins)]-I(ins);
                break;
            case 0x28: /* l.andi */
                if(verbose&1)printf("l.andi   r%-2i, r%-2i, %7i    r%-2i=%08x r%-2i=%08x\n",D(ins),A(ins),
                    		I(ins),D(ins),gpr[D(ins)],A(ins),gpr[A(ins)]);
                gpr[D(ins)]=gpr[A(ins)] & K(ins);
                break;
            case 0x29: /* l.ori */
                if(verbose&1)printf("l.ori    r%-2i, r%-2i, %7i    r%-2i=%08x r%-2i=%08x\n",D(ins),A(ins),
                    		I(ins),D(ins),gpr[D(ins)],A(ins),gpr[A(ins)]);
                gpr[D(ins)]=gpr[A(ins)] | K(ins);
                break;
            case 0x2A: /* l.xori */
                if(verbose&1)printf("l.xori   r%-2i, r%-2i, %7i    r%-2i=%08x r%-2i=%08x\n",D(ins),A(ins),
                    		I(ins),D(ins),gpr[D(ins)],A(ins),gpr[A(ins)]);
                gpr[D(ins)]=gpr[A(ins)] ^ I(ins);
                break;
            case 0x2B: /* l.muli */
                if(verbose&1)printf("l.muli   r%-2i, r%-2i, %7i    r%-2i=%08x r%-2i=%08x\n",D(ins),A(ins),
                    		I(ins),D(ins),gpr[D(ins)],A(ins),gpr[A(ins)]);
                gpr[D(ins)]=(int)gpr[A(ins)] * I(ins);
                break;
                
            case 0x2D: /* shifts: */
                switch(ins&0xe0) {
                    case 0: /* l.slli */
                        if(verbose&1)printf("l.slli   r%-2i, r%-2i, %7i    r%-2i=%08x r%-2i=%08x\n",D(ins),A(ins),
                            I(ins),D(ins),gpr[D(ins)],A(ins),gpr[A(ins)]);
                        gpr[D(ins)]=gpr[A(ins)] << L(ins);
                        break;
                    case 0x20: /* l.srli */
                        if(verbose&1)printf("l.srli   r%-2i, r%-2i, %7i    r%-2i=%08x r%-2i=%08x\n",D(ins),A(ins),
                            I(ins),D(ins),gpr[D(ins)],A(ins),gpr[A(ins)]);
                        gpr[D(ins)]=gpr[A(ins)] >> L(ins);
                        break;
                    case 0x40: /* l.srlai */
                        if(verbose&1)printf("l.srlai  r%-2i, r%-2i, %7i    r%-2i=%08x r%-2i=%08x\n",D(ins),A(ins),
                            I(ins),D(ins),gpr[D(ins)],A(ins),gpr[A(ins)]);
                        gpr[D(ins)]=(gpr[A(ins)] >> L(ins)) | (gpr[A(ins)]&0x80000000? POW2(L(ins)) :0);
                        break;
                    case 0x80: /* l.rori */
                        if(verbose&1)printf("l.rori   r%-2i, r%-2i, %7i    r%-2i=%08x r%-2i=%08x\n",D(ins),A(ins),
                            I(ins),D(ins),gpr[D(ins)],A(ins),gpr[A(ins)]);
                        gpr[D(ins)] = (gpr[A(ins)] >> L(ins)) |
                            (gpr[A(ins)] << (31-L(ins)));
                }
                break;

            case 0x2E: /* l.sf*i */
                switch(D(ins)) {
                    case 0: /* l.sfeqi */
                        if(verbose&1)printf("l.sfeqi  r%-2i, %7i    r%-2i=%08x\n",A(ins),
                            I(ins),A(ins),gpr[A(ins)]);
                        flags=gpr[A(ins)] == I(ins);
                        break;
                    case 1: /* l.sfnei */
                        if(verbose&1)printf("l.sfnei  r%-2i, %7i    r%-2i=%08x\n",A(ins),
                            I(ins),A(ins),gpr[A(ins)]);
                        flags=gpr[A(ins)] != I(ins);
                        break;
                    case 2: /* l.sfgtui */
                        if(verbose&1)printf("l.sfgtui r%-2i, %7i    r%-2i=%08x\n",A(ins),
                            I(ins),A(ins),gpr[A(ins)]);
                        flags=gpr[A(ins)] > I(ins);
                        break;
                    case 3: /* l.sfgeui */
                        if(verbose&1)printf("l.sfgeui r%-2i, %7i    r%-2i=%08x\n",A(ins),
                            I(ins),A(ins),gpr[A(ins)]);
                        flags=gpr[A(ins)] >= I(ins);
                        break;
                    case 4: /* l.sfltui */
                        if(verbose&1)printf("l.sfltui r%-2i, %7i    r%-2i=%08x\n",A(ins),
                            I(ins),A(ins),gpr[A(ins)]);
                        flags=gpr[A(ins)] < I(ins);
                        break;
                    case 5: /* l.sfleui */
                        if(verbose&1)printf("l.sfleui r%-2i, %7i    r%-2i=%08x\n",A(ins),
                            I(ins),A(ins),gpr[A(ins)]);
                        flags=gpr[A(ins)] <= I(ins);
                        break;
                    case 10: /* l.sfgtsi */
                        if(verbose&1)printf("l.sfgtsi r%-2i, %7i    r%-2i=%08x\n",A(ins),
                            I(ins),A(ins),gpr[A(ins)]);
                        flags=(int)gpr[A(ins)] > I(ins);
                        break;
                    case 11: /* l.sfgesi */
                        if(verbose&1)printf("l.sfgesi r%-2i, %7i    r%-2i=%08x\n",A(ins),
                            I(ins),A(ins),gpr[A(ins)]);
                        flags=(int)gpr[A(ins)] >= I(ins);
                        break;
                    case 12: /* l.sfltsi */
                        if(verbose&1)printf("l.sfltsi r%-2i, %7i    r%-2i=%08x\n",A(ins),
                            I(ins),A(ins),gpr[A(ins)]);
                        flags=(int)gpr[A(ins)] < I(ins);
                        break;
                    case 13: /* l.sflesi */
                        if(verbose&1)printf("l.sflesi r%-2i, %7i    r%-2i=%08x\n",A(ins),
                            I(ins),A(ins),gpr[A(ins)]);
                        flags=(int)gpr[A(ins)] <= I(ins);
                        break;
                }
                break;
                
            case 0x35: /* l.sw */
                if(verbose&1)printf("l.sw     %7i(r%-2i),r%-2i     r%-2i=%08x r%-2i=%08x\n",SI(ins),A(ins),
                    	B(ins),A(ins),gpr[A(ins)],B(ins),gpr[B(ins)]);
                SMEM(gpr[A(ins)] + SI(ins), gpr[B(ins)]);
                break;
            case 0x36: /* l.sb */
                if(verbose&1)printf("l.sb     %7i(r%-2i),r%-2i     r%-2i=%08x r%-2i=%08x\n",SI(ins),A(ins),
                    	B(ins),A(ins),gpr[A(ins)],B(ins),gpr[B(ins)]);
                SMEM8(gpr[A(ins)] + SI(ins), gpr[B(ins)]);
                break;
            case 0x37: /* l.sh */
                if(verbose&1)printf("l.sh     %7i(r%-2i),r%-2i     r%-2i=%08x r%-2i=%08x\n",SI(ins),A(ins),
                    	B(ins),A(ins),gpr[A(ins)],B(ins),gpr[B(ins)]);
                SMEM16(gpr[A(ins)] + SI(ins), gpr[B(ins)]);
                break;
                
            
            case 0x38: /*  */
                switch(ins&0xcf) {
                    case 0: /* l.add */
                        if(verbose&1)printf("l.add    r%-2i, r%-2i, r%-2i"
                           "        r%-2i=%08x r%-2i=%08x r%-2i=%08x\n",
                           D(ins),A(ins),B(ins),D(ins),gpr[D(ins)],A(ins),gpr[A(ins)],B(ins),gpr[B(ins)]);
                        gpr[D(ins)]=gpr[A(ins)] + gpr[B(ins)];
                        break;
                    case 1: /* l.addc */
                        if(verbose&1)printf("l.addc   r%-2i, r%-2i, r%-2i"
                           "        r%-2i=%08x r%-2i=%08x r%-2i=%08x carry=%i\n",
                           D(ins),A(ins),B(ins),D(ins),gpr[D(ins)],A(ins),gpr[A(ins)],B(ins),gpr[B(ins)],carry);
                        gpr[D(ins)]=gpr[A(ins)] + gpr[B(ins)] + carry;
                        break;
                    case 2: /* l.sub */
                        if(verbose&1)printf("l.sub    r%-2i, r%-2i, r%-2i"
                           "        r%-2i=%08x r%-2i=%08x r%-2i=%08x\n",
                           D(ins),A(ins),B(ins),D(ins),gpr[D(ins)],A(ins),gpr[A(ins)],B(ins),gpr[B(ins)]);
                        gpr[D(ins)]=gpr[A(ins)] - gpr[B(ins)];
                        break;
                    case 3: /* l.and */
                        if(verbose&1)printf("l.and    r%-2i, r%-2i, r%-2i"
                           "        r%-2i=%08x r%-2i=%08x r%-2i=%08x\n",
                           D(ins),A(ins),B(ins),D(ins),gpr[D(ins)],A(ins),gpr[A(ins)],B(ins),gpr[B(ins)]);
                        gpr[D(ins)]=gpr[A(ins)] & gpr[B(ins)];
                        break;
                    case 4: /* l.or */
                        if(verbose&1)printf("l.or     r%-2i, r%-2i, r%-2i"
                           "        r%-2i=%08x r%-2i=%08x r%-2i=%08x\n",
                           D(ins),A(ins),B(ins),D(ins),gpr[D(ins)],A(ins),gpr[A(ins)],B(ins),gpr[B(ins)]);
                        gpr[D(ins)]=gpr[A(ins)] | gpr[B(ins)];
                        break;
                    case 5: /* l.xor */
                        if(verbose&1)printf("l.xor    r%-2i, r%-2i, r%-2i"
                           "        r%-2i=%08x r%-2i=%08x r%-2i=%08x\n",
                           D(ins),A(ins),B(ins),D(ins),gpr[D(ins)],A(ins),gpr[A(ins)],B(ins),gpr[B(ins)]);
                        gpr[D(ins)]=gpr[A(ins)] ^ gpr[B(ins)];
                        break;
                    case 0x0b: /* l.mulu */
                        if(verbose&1)printf("l.mulu   r%-2i, r%-2i, r%-2i"
                           "        r%-2i=%08x r%-2i=%08x r%-2i=%08x\n",
                           D(ins),A(ins),B(ins),D(ins),gpr[D(ins)],A(ins),gpr[A(ins)],B(ins),gpr[B(ins)]);
                        gpr[D(ins)]=gpr[A(ins)] * gpr[B(ins)];
                        break;
                    case 0x06: /* l.mul */
                        if(verbose&1)printf("l.mul    r%-2i, r%-2i, r%-2i"
                           "        r%-2i=%08x r%-2i=%08x r%-2i=%08x\n",
                           D(ins),A(ins),B(ins),D(ins),gpr[D(ins)],A(ins),gpr[A(ins)],B(ins),gpr[B(ins)]);
                        gpr[D(ins)]=(int)gpr[A(ins)] * (int)gpr[B(ins)];
                        break;
                    case 7: /* l.mac */
                        if(verbose&1)printf("l.mac    r%-2i, r%-2i, r%-2i"
                           "        r%-2i=%08x r%-2i=%08x r%-2i=%08x\n",
                           D(ins),A(ins),B(ins),D(ins),gpr[D(ins)],A(ins),gpr[A(ins)],B(ins),gpr[B(ins)]);
                        mac += (int)gpr[A(ins)] * (int)gpr[B(ins)];
                    case 9: /* l.div */
                        if(verbose&1)printf("l.div    r%-2i, r%-2i, r%-2i"
                           "        r%-2i=%08x r%-2i=%08x r%-2i=%08x\n",
                           D(ins),A(ins),B(ins),D(ins),gpr[D(ins)],A(ins),gpr[A(ins)],B(ins),gpr[B(ins)]);
                        if(gpr[B(ins)]) {
                            gpr[D(ins)]=(int)gpr[A(ins)] / (int)gpr[B(ins)];
                        } else {
                            /* XXX exception */
                            div0=1;
                        }
                        break;
                    case 0xa: /* l.divu */
                        if(verbose&1)printf("l.divu   r%-2i, r%-2i, r%-2i"
                           "        r%-2i=%08x r%-2i=%08x r%-2i=%08x\n",
                           D(ins),A(ins),B(ins),D(ins),gpr[D(ins)],A(ins),gpr[A(ins)],B(ins),gpr[B(ins)]);
                        if(gpr[B(ins)]) {
                            gpr[D(ins)]=gpr[A(ins)] / gpr[B(ins)];
                        } else {
                            /* XXX exception */
                            div0=1;
                        }
                        break;
                    case 8: /* shifts */
                        switch(ins&0xf0) {
                            case 0: /* l.sll */
                                if(verbose&1)printf("l.sll    r%-2i, r%-2i, r%-2i"
                                     "        r%-2i=%08x r%-2i=%08x r%-2i=%08x\n",
                                     D(ins),A(ins),B(ins),D(ins),gpr[D(ins)],
                                     A(ins),gpr[A(ins)],B(ins),gpr[B(ins)]);
                                gpr[D(ins)]=gpr[A(ins)] << gpr[B(ins)];
                                break;
/* XXX document differs from binutils  srl - srla*/
                            case 0x20: /* l.srl */
                                if(verbose&1)printf("l.srl    r%-2i, r%-2i, r%-2i"
                                     "        r%-2i=%08x r%-2i=%08x r%-2i=%08x\n",
                                     D(ins),A(ins),B(ins),D(ins),gpr[D(ins)],
                                     A(ins),gpr[A(ins)],B(ins),gpr[B(ins)]);
                                gpr[D(ins)]=gpr[A(ins)] >> gpr[B(ins)];
                                break;
                            case 0x10: /* l.srla */
                                if(verbose&1)printf("l.srla   r%-2i, r%-2i, r%-2i"
                                     "        r%-2i=%08x r%-2i=%08x r%-2i=%08x\n",
                                     D(ins),A(ins),B(ins),D(ins),gpr[D(ins)],
                                     A(ins),gpr[A(ins)],B(ins),gpr[B(ins)]);
                                gpr[D(ins)]=(gpr[A(ins)] >> gpr[B(ins)]) | 
                                    (gpr[A(ins)]&0x80000000? POW2(gpr[B(ins)]) :0);
                                break;
                            case 0x30: /* l.ror */
                                if(verbose&1)printf("l.ror    r%-2i, r%-2i, r%-2i"
                                     "        r%-2i=%08x r%-2i=%08x r%-2i=%08x\n",
                                     D(ins),A(ins),B(ins),D(ins),gpr[D(ins)],
                                     A(ins),gpr[A(ins)],B(ins),gpr[B(ins)]);
                                gpr[D(ins)] = (gpr[A(ins)] >> gpr[B(ins)]) |
                                    (gpr[A(ins)] << (31-gpr[B(ins)]));
                        }
                        break;
                        
                    case 0x0c: /* l.exths */
                        gpr[D(ins)]= SEX16(gpr[A(ins)]);
                        break;
                    case 0x4c: /* l.extbs */
                        gpr[D(ins)]=SEX8(gpr[A(ins)]);
                        break;
                    case 0x8c: /* l.exthz */
                        gpr[D(ins)]= gpr[A(ins)] & 0xffff;
                        break;
                    case 0xcc: /* l.extbz */
                        gpr[D(ins)]= gpr[A(ins)] & 0xff;
                        break;
                }
                break;
                
            case 0x39: /* l.sf* */
                switch(D(ins)) {
                    case 0: /* l.sfeq */
                        if(verbose&1)printf("l.sfeq   r%-2i, r%-2i"
                             "             r%-2i=%08x r%-2i=%08x\n",
                             A(ins),B(ins),A(ins),gpr[A(ins)],B(ins),gpr[B(ins)]);
                        flags=gpr[A(ins)] == gpr[B(ins)];
                        break;
                    case 1: /* l.sfne */
                        if(verbose&1)printf("l.sfne   r%-2i, r%-2i"
                             "             r%-2i=%08x r%-2i=%08x\n",
                             A(ins),B(ins),A(ins),gpr[A(ins)],B(ins),gpr[B(ins)]);
                        flags=gpr[A(ins)] != gpr[B(ins)];
                        break;
                    case 2: /* l.sfgtu */
                        if(verbose&1)printf("l.sfgtu  r%-2i, r%-2i"
                             "             r%-2i=%08x r%-2i=%08x\n",
                             A(ins),B(ins),A(ins),gpr[A(ins)],B(ins),gpr[B(ins)]);
                        flags=gpr[A(ins)] > gpr[B(ins)];
                        break;
                    case 3: /* l.sfgeu */
                        if(verbose&1)printf("l.sfgeu  r%-2i, r%-2i"
                             "             r%-2i=%08x r%-2i=%08x\n",
                             A(ins),B(ins),A(ins),gpr[A(ins)],B(ins),gpr[B(ins)]);
                        flags=gpr[A(ins)] >= gpr[B(ins)];
                        break;
                    case 4: /* l.sfltu */
                        if(verbose&1)printf("l.sfltu  r%-2i, r%-2i"
                             "             r%-2i=%08x r%-2i=%08x\n",
                             A(ins),B(ins),A(ins),gpr[A(ins)],B(ins),gpr[B(ins)]);
                        flags=gpr[A(ins)] < gpr[B(ins)];
                        break;
                    case 5: /* l.sfleu */
                        if(verbose&1)printf("l.sfleu  r%-2i, r%-2i"
                             "             r%-2i=%08x r%-2i=%08x\n",
                             A(ins),B(ins),A(ins),gpr[A(ins)],B(ins),gpr[B(ins)]);
                        flags=gpr[A(ins)] <= gpr[B(ins)];
                        break;
                    case 10: /* l.sfgts */
                        if(verbose&1)printf("l.sfgts  r%-2i, r%-2i"
                             "             r%-2i=%08x r%-2i=%08x\n",
                             A(ins),B(ins),A(ins),gpr[A(ins)],B(ins),gpr[B(ins)]);
                        flags=(int)gpr[A(ins)] > (int)gpr[B(ins)];
                        break;
                    case 11: /* l.sfges */
                        if(verbose&1)printf("l.sfges  r%-2i, r%-2i"
                             "             r%-2i=%08x r%-2i=%08x\n",
                             A(ins),B(ins),A(ins),gpr[A(ins)],B(ins),gpr[B(ins)]);
                        flags=(int)gpr[A(ins)] >= (int)gpr[B(ins)];
                        break;
                    case 12: /* l.sflts */
                        if(verbose&1)printf("l.sflts  r%-2i, r%-2i"
                             "             r%-2i=%08x r%-2i=%08x\n",
                             A(ins),B(ins),A(ins),gpr[A(ins)],B(ins),gpr[B(ins)]);
                        flags=(int)gpr[A(ins)] < (int)gpr[B(ins)];
                        break;
                    case 13: /* l.sfles */
                        if(verbose&1)printf("l.sfles  r%-2i, r%-2i"
                             "             r%-2i=%08x r%-2i=%08x\n",
                             A(ins),B(ins),A(ins),gpr[A(ins)],B(ins),gpr[B(ins)]);
                        flags=(int)gpr[A(ins)] <= (int)gpr[B(ins)];
                        break;
                }
                break;
            
            default:
                /* XXX illegal instruction */
                printf("Illegal instruction %08x : %08x\n", pc, ins);
                exit(17);
                break;
                
        }
        if(++nextjump) {
            pc+=4;
        } else {
            pc=newpc;
        }
        
    }
    return 0;
}
