#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <getopt.h>
#include <byteswap.h>

/* sign extending */
#define SEX8(x) (x | ((x&0x80)?0xffffff00:0))
#define SEX16(x) (x | ((x&0x8000)?0xffff0000:0))
#define SEX26(x) (x | ((x&0x2000000)?0xfc000000:0))
#define SEX28(x) (x | ((x&0x8000000)?0xf0000000:0))

/* bit fields in instruction words */
#define OP(ins) ((ins>>26) & 0x3f)
#define EOP(ins) ((ins>>16) & 0xffff)
#define N(ins) SEX26(ins&0x3ffffff)
#define I(ins) SEX16(ins&0xffff)
#define SI(ins) SEX16((ins&0x7ff)+((ins&0x3e00000)>>10))
#define SK(ins) ((ins&0x7ff)+((ins&0x3e00000)>>10))

#define K(ins) (ins&0xffff)
#define L(ins) (ins&0x3f)
#define B(ins) ((ins>>11)&0x1f)
#define A(ins) ((ins>>16)&0x1f)
#define D(ins) ((ins>>21)&0x1f)

/* link register is a special register */
#define _lr gpr[11]

#define POW(i) (1<<i)
#define POW1(i) ((1<<i)-1)
#define POW2(i) (~((1<<(32-i))-1))

unsigned int gpr[16];
unsigned int pc, flags, carry, overflow;

unsigned int VR = 0x12008000;
unsigned int UPR = 0x703f;
unsigned int SR;

unsigned int EPCR[16];
unsigned int EEAR[16];
unsigned int ESR[16];
unsigned int CCR[16];

unsigned int DTLBMR[256];
unsigned int DTLBTR[256];
unsigned int DMMUCR1;
unsigned int DMMUCR2;

unsigned int ITLBMR[256];
unsigned int ITLBTR[256];
unsigned int IMMUCR1;
unsigned int IMMUCR2;

long long MAC;

unsigned int TTCR;

int tickte, tickie;
unsigned int inscounter;
long long realinscounter;
int nextjump;
unsigned int newpc;
unsigned int forceex;

#define SR_SUPV	(1<<0)
#define SR_EXR	(1<<1)
#define SR_IME	(1<<6)
#define SR_DME	(1<<5)

#define EX_RST  0x100
#define EX_DFPE 0x300
#define EX_IFPE 0x400
#define EX_LPI	0x500
#define EX_ALGN	0x600
#define EX_ILL	0x700
#define EX_HPI	0x800
#define EX_DTLB	0x900
#define EX_ITLB	0xa00
#define EX_SYS	0xc00
#define EX_BRK	0xd00

#include "memory.h"

void reset() {
    TTCR=0;
    tickte=0;
    tickie=0;
    inscounter=0;    
}

int exception(unsigned int type, unsigned int ea) {
    forceex=0;
    switch(type){
        case EX_LPI:
        case EX_HPI:
        case EX_DTLB:
        case EX_ITLB:
            EPCR[0]=pc;
            break;
        case EX_ALGN:
        case EX_ILL:
        case EX_SYS:
        case EX_BRK:
        default:
            EPCR[0]=pc+4;
    }
    EEAR[0]=ea;
    ESR[0]=SR;
    SR |= SR_SUPV;
    SR &= ~(SR_EXR | SR_IME | SR_DME);
    
    newpc=type;
    nextjump=-1;
    return 0;
}

int usage(char *s) {
    fprintf(stderr, "%s\n",s);
    return 0;
}

int readfile(char *name, int address) {
    int f,i,j;

    f=open(name, O_RDONLY);
    i=read(f, &ram[address], 1024*1024);
#ifdef BSWAP
    if(i>0) {
        for(j=0;j<i;j+=4) {
            unsigned int k;
            k=*(unsigned int *)(ram+j+address);
            ram[j+address]=k>>24;
            ram[j+1+address]=(k>>16)&0xff;
            ram[j+2+address]=(k>>8)&0xff;
            ram[j+3+address]=k&0xff;
        }
    }
#endif
    close(f);
    return 0;
}

int main(int argc, char *argv[]) {
    int c, f, i, j, ex; 
    int verbose;

    verbose=0;
    for (;;) {
        if (-1 == (c = getopt(argc, argv, "hvw")))
            break;
        switch (c) {
        case 'v':
            verbose=1;
            break;
        case 'w':
            verbose=2;
            break;
        case 'h':
        default:
            usage(argv[0]);
            exit(1);
        }
    }

    if (optind == argc) {
        usage(argv[0]);
        exit(1);
    }

    readfile("rom.or32",0);
    readfile(argv[optind],64*1024);

    nextjump=0;
    forceex=0;
    
    realinscounter=0;

    reset();

    ex=0;
    SR=1;
    pc=EX_RST;
    gpr[0]=0;

    while(1) {
        unsigned int ins, op;
        
        ins=IMEM(pc);
        if(forceex) {
            exception(forceex,pc);
        } else {
            realinscounter++;
            if(tickte) {
                inscounter++;
                if(inscounter==(TTCR&0xfffffff)) {
                    if(tickie) {
                        TTCR |= 0x10000000;
                        exception(EX_HPI, 0);
                        ex=1;
                    }
                    if(TTCR&0x40000000) {
                        inscounter=0;
                    } else {
                        TTCR &= ~0x80000000;
                        tickte=0;
                    }
                }
            }
            if(ex) {
                ex=0;
            } else {
                if(verbose&1)printf("%8x: %08x   ", pc, ins);
                if(verbose&2)printf("%8x: %08x\n", pc, ins);
                op=OP(ins);
                switch(op) {
                    case 0x00: /* l.j */
                        if(verbose&1)printf("l.j      %7i\n",N(ins));
                        newpc=pc + (N(ins)<<2);
                        nextjump=-2;
                        break;
                    case 0x01: /* l.jal */
                        if(verbose&1)printf("l.jal    %7i\n",N(ins));
                        _lr=pc+8;
                        newpc= pc + (N(ins)<<2);
                        nextjump=-2;
                        break;
                    case 0x03: /* l.bnf */
                        if(verbose&1)printf("l.bnf    %7i              f=%i\n",N(ins),flags&1);
                        if(!(flags&1)) {
                            nextjump=-2;
                            newpc= pc + (N(ins)<<2);
                        }
                        break;
                    case 0x04: /* l.bf */
                        if(verbose&1)printf("l.bf     %7i              f=%i\n",N(ins),flags&1);
                        if(flags&1) { 
                            nextjump=-2;
                            newpc= pc + (N(ins)<<2);
                        }
                        break;
                    case 0x05: /* l.nop */
                        if(verbose&1)printf("l.nop\n");
                        /* XXX only if top 8 bits is 0x15 */
                        break;
    
                    case 0x06: /* l.movhi */
                        if(verbose&1)printf("l.movhi  r%-2i,%4x\n",D(ins),K(ins));
                        gpr[D(ins)] = K(ins)<<16;
                        break;
    
                    case 0x07: /* l.mfspr */
                        if(verbose&1)printf("l.mfspr  r%-2i, r%-2i, %7i    r%-2i=%08x r%-2i=%08x\n",D(ins),
                    	        A(ins),K(ins),D(ins),gpr[D(ins)],A(ins),gpr[A(ins)]);
                        switch(SR&1) {
                            case 1:
                                switch(K(ins)+gpr[A(ins)]) {
                                    case 1: /* VR */
                                        gpr[D(ins)]=VR;
                                        break;
                                    case 2: /* UPR */
                                        gpr[D(ins)]=UPR;
                                        break;
                                    case 3: /* SR */
                                        gpr[D(ins)]=SR;
                                        break;
                                    case 16 ... 31:
                                        gpr[D(ins)]=EPCR[K(ins)+gpr[A(ins)]-16];
                                        break;
                                    case 48 ... 63:
                                        gpr[D(ins)]=EEAR[K(ins)+gpr[A(ins)]-48];
                                        break;
                                    case 64 ... 79:
                                        gpr[D(ins)]=ESR[K(ins)+gpr[A(ins)]-64];
                                        break;
                                    case 80 ... 95:
                                        gpr[D(ins)]=CCR[K(ins)+gpr[A(ins)]-80];
                                        break;
                                    case 2048 + 512:
                                        gpr[D(ins)] = DMMUCR1;
                                        break;
                                    case 2048 + 513:
                                        gpr[D(ins)] = DMMUCR2;
                                        break;
                                    case 4096 + 512:
                                        gpr[D(ins)] = IMMUCR1;
                                        break;
                                    case 4096 + 513:
                                        gpr[D(ins)] = IMMUCR2;
                                        break;
    
                                    case 5*2048:
                                        gpr[D(ins)] = MAC & 0xffffffff;
                                        break;
                                    case 5*2048 + 1:
                                        gpr[D(ins)] = (MAC>>32) & 0xffffffff;
                                        break;
    
                                    case 10*2048:
                                        gpr[D(ins)] = TTCR;
                                        TTCR &= ~0x10000000;
                                        break;
                                    case 10*2048 + 1:
                                        gpr[D(ins)] = inscounter;
                                        break;
                                }
                                break;
                            case 0:
                                switch(K(ins)+gpr[A(ins)]) {
                                    case 5*2048:
                                        gpr[D(ins)] = MAC & 0xffffffff;
                                        break;
                                    case 5*2048 + 1:
                                        gpr[D(ins)] = (MAC>>32) & 0xffffffff;
                                        break;
    
                                    case 10*2048 + 1:
                                        gpr[D(ins)] = inscounter;
                                        break;
                                }
                                break;
                        }                
                        break;
                    case 0x08:
                        switch(EOP(ins)) {
                            case 0x2000: /* sys K(ins) */
                                if(verbose&1)printf("l.sys    %4x\n",K(ins));
                                    switch(K(ins)) {
                                        case 201:
                                            gpr[3]=realinscounter;
                                            break;
                                        case 203:
                                            printf("%Li instructions.\n",realinscounter);
                                            return 0;
                                            break;
                                        default:
                                            exception(EX_SYS, 0);
                                    }
                                break;
                            case 0x2100: /* l.brk */
                                exception(EX_BRK, 0);
                                break;
                            case 0x2200: /* l.msync */
                                break;
                            case 0x2300: /* l.csync */
                                break;
                            case 0x2280: /* l.psync */
                                break;
                        
                        }
                        break;
                    case 0x09: /* l.rfe */
                        SR=ESR[0];
                        newpc=EPCR[0];
                        nextjump=-1;
                        break;
    
                    case 0x10: /* l.mtspr */
                        if(verbose&1)printf("l.mtspr  r%-2i, r%-2i, %7i    r%-2i=%08x r%-2i=%08x\n",A(ins),
                    	        B(ins),SK(ins),A(ins),gpr[A(ins)],B(ins),gpr[B(ins)]);
                        switch(SR&1) {
                            case 1:
                                switch(SK(ins)+gpr[A(ins)]) {
                                    case 3: /* SR */
                                        SR=gpr[B(ins)];
                                        break;
                                    case 16 ... 31:
                                        EPCR[K(ins)+gpr[A(ins)]-16]=gpr[B(ins)];
                                        break;
                                    case 48 ... 63:
                                        EEAR[K(ins)+gpr[A(ins)]-48]=gpr[B(ins)];
                                        break;
                                    case 64 ... 79:
                                        ESR[K(ins)+gpr[A(ins)]-64]=gpr[B(ins)];
                                        break;
                                    case 80 ... 95:
                                        CCR[K(ins)+gpr[A(ins)]-80]=gpr[B(ins)];
                                        break;
                        
                                    case 2048+0 ... 2048+255: /* DTLBMR */
                                        DTLBMR[SK(ins)+gpr[A(ins)]-2048]=gpr[B(ins)];
                                        break;
                                    case 2048+256 ... 2048+511: /* DTLBTR */
                                        DTLBTR[SK(ins)+gpr[A(ins)]-2048-256]=gpr[B(ins)];
                                        break;                        
                                    case 2048+512: /* DMMUCR1 */
                                        DMMUCR1=gpr[B(ins)];
                                        break;
                                    case 2048+513: /* DMMUCR2 */
                                        DMMUCR2=gpr[B(ins)];
                                        break;
    
                                    case 4096+0 ... 4096+255: /* ITLBMR */
                                        ITLBMR[SK(ins)+gpr[A(ins)]-4096]=gpr[B(ins)];
                                        break;
                                    case 4096+256 ... 4096+511: /* ITLBTR */
                                        ITLBTR[SK(ins)+gpr[A(ins)]-4096-256]=gpr[B(ins)];
                                        break;
                                    case 4096+512: /* IMMUCR1 */
                                        IMMUCR1=gpr[B(ins)];
                                        break;
                                    case 4096+513: /* IMMUCR2 */
                                        IMMUCR2=gpr[B(ins)];
                                        break;
    
                                    case 5*2048:
                                        MAC &= 0xffffffff00000000;
                                        MAC |= gpr[B(ins)];
                                        break;
                                    case 5*2048 + 1:
                                        MAC &= 0xffffffff;
                                        MAC |= (long long)gpr[B(ins)]<<32;
                                        break;
    
                                    case 10*2048:
                                        TTCR = gpr[B(ins)];
                                        tickte = TTCR&0x80000000;
                                        tickie = TTCR&0x20000000;
                                        if(TTCR&0x40000000) {
                                            inscounter=0;
                                        }
                                        break;
                                    case 10*2048 + 1:
                                        inscounter = gpr[B(ins)];
                                        break;
                                }
                                break;
                            case 0:
                                switch(SK(ins)+gpr[A(ins)]) {
                                    case 5*2048:
                                        MAC &= 0xffffffff00000000;
                                        MAC |= gpr[B(ins)];
                                        break;
                                    case 5*2048 + 1:
                                        MAC &= 0xffffffff;
                                        MAC |= (long long)gpr[B(ins)]<<32;
                                        break;
                                }
                                break;
                        };
                        break;
                    case 0x11: /* l.jr */
                        if(verbose&1)printf("l.jr     r%-2i                  r%-2i=%08x\n",B(ins),B(ins),gpr[B(ins)]);
                        nextjump=-2;
                        newpc=gpr[B(ins)];
                        break;
                    case 0x12: /* l.jalr */
                        if(verbose&1)printf("l.jlar   r%-2i                  r%-2i=%08x\n",B(ins),B(ins),gpr[B(ins)]);
                        nextjump=-2;
                        newpc=gpr[B(ins)];
                        _lr=pc+8;
                        break;
                
                    case 0x22: /* l.lws */
                        if(verbose&1)printf("l.lws    %7i(r%-2i),r%-2i     r%-2i=%08x mem=%08x\n",I(ins),A(ins),
                    		        D(ins),A(ins),gpr[A(ins)],MEM(gpr[A(ins)]+I(ins)));
                        gpr[D(ins)]=MEM(gpr[A(ins)]+I(ins));
                        if(forceex)exception(forceex,pc);
                        break;
                    case 0x21: /* l.lwz */
                        if(verbose&1)printf("l.lwz    %7i(r%-2i),r%-2i     r%-2i=%08x mem=%08x\n",I(ins),A(ins),
                    		        D(ins),A(ins),gpr[A(ins)],MEM(gpr[A(ins)]+I(ins)));
                        gpr[D(ins)]=MEM(gpr[A(ins)]+I(ins));
                        if(forceex)exception(forceex,pc);
                        break;
                    case 0x23: /* l.lbz */
                        if(verbose&1)printf("l.lbz    %7i(r%-2i),r%-2i     r%-2i=%08x mem=%02x\n",I(ins),A(ins),
                    		        D(ins),A(ins),gpr[A(ins)],MEM8(gpr[A(ins)]+I(ins)));
                        gpr[D(ins)]=MEM8(gpr[A(ins)]+I(ins));
                        if(forceex)exception(forceex,pc);
                        break;
                    case 0x24: /* l.lbs */
                        if(verbose&1)printf("l.lbs    %7i(r%-2i),r%-2i     r%-2i=%08x mem=%02x\n",I(ins),A(ins),
                    		        D(ins),A(ins),gpr[A(ins)],MEM8(gpr[A(ins)]+I(ins)));
                        gpr[D(ins)]=SEX8(MEM8(gpr[A(ins)]+I(ins)));
                        if(forceex)exception(forceex,pc);
                        break;
                    case 0x25: /* l.lhz */
                        if(verbose&1)printf("l.lhz    %7i(r%-2i),r%-2i     r%-2i=%08x mem=%04x\n",I(ins),A(ins),
                    		        D(ins),A(ins),gpr[A(ins)],MEM16(gpr[A(ins)]+I(ins)));
                        gpr[D(ins)]=MEM16(gpr[A(ins)]+I(ins));
                        if(forceex)exception(forceex,pc);
                        break;
                    case 0x26: /* l.lhs */
                        if(verbose&1)printf("l.lhs    %7i(r%-2i),r%-2i     r%-2i=%08x mem=%04x\n",I(ins),A(ins),
                    		        D(ins),A(ins),gpr[A(ins)],MEM16(gpr[A(ins)]+I(ins)));
                        gpr[D(ins)]=SEX16(MEM16(gpr[A(ins)]+I(ins)));
                        if(forceex)exception(forceex,pc);
                        break;
                
                    case 0x27: /* l.addi */
                        if(verbose&1)printf("l.addi   r%-2i, r%-2i, %7i    r%-2i=%08x r%-2i=%08x\n",D(ins),A(ins),
                    		        I(ins),D(ins),gpr[D(ins)],A(ins),gpr[A(ins)]);
                        gpr[D(ins)]=gpr[A(ins)]+I(ins);
                        carry=gpr[D(ins)]<gpr[A(ins)];
                        break;
                    case 0x29: /* l.andi */
                        if(verbose&1)printf("l.andi   r%-2i, r%-2i, %7i    r%-2i=%08x r%-2i=%08x\n",D(ins),A(ins),
                    		        I(ins),D(ins),gpr[D(ins)],A(ins),gpr[A(ins)]);
                        gpr[D(ins)]=gpr[A(ins)] & K(ins);
                        break;
                    case 0x2a: /* l.ori */
                        if(verbose&1)printf("l.ori    r%-2i, r%-2i, %7i    r%-2i=%08x r%-2i=%08x\n",D(ins),A(ins),
                    		        I(ins),D(ins),gpr[D(ins)],A(ins),gpr[A(ins)]);
                        gpr[D(ins)]=gpr[A(ins)] | K(ins);
                        break;
                    case 0x2b: /* l.xori */
                        if(verbose&1)printf("l.xori   r%-2i, r%-2i, %7i    r%-2i=%08x r%-2i=%08x\n",D(ins),A(ins),
                    		        I(ins),D(ins),gpr[D(ins)],A(ins),gpr[A(ins)]);
                        gpr[D(ins)]=gpr[A(ins)] ^ I(ins);
                        break;
                    case 0x2c: /* l.muli */
                        if(verbose&1)printf("l.muli   r%-2i, r%-2i, %7i    r%-2i=%08x r%-2i=%08x\n",D(ins),A(ins),
                    		        I(ins),D(ins),gpr[D(ins)],A(ins),gpr[A(ins)]);
                        gpr[D(ins)]=(int)gpr[A(ins)] * I(ins);
                        break;
                
                    case 0x2d: /* l.maci */
    		        if(verbose&1)printf("l.maci    r%-2i, r%-2i, %7i"
                    	        "        r%-2i=%08x r%-2i=%08x\n",
                                   D(ins),A(ins),I(ins),D(ins),gpr[D(ins)],A(ins),gpr[A(ins)]);
                        MAC += (int)gpr[A(ins)] * I(ins);
                        break;           
                    case 0x2e: /* shifts: */
                        switch(ins&0x1c0) {
                            case 0: /* l.slli */
                                if(verbose&1)printf("l.slli   r%-2i, r%-2i, %7i    r%-2i=%08x r%-2i=%08x\n",
                                    D(ins),A(ins),
                                    L(ins),D(ins),gpr[D(ins)],A(ins),gpr[A(ins)]);
                                gpr[D(ins)]=gpr[A(ins)] << L(ins);
                                break;
                            case 0x40: /* l.srli */
                                if(verbose&1)printf("l.srli   r%-2i, r%-2i, %7i    r%-2i=%08x r%-2i=%08x\n",
        			    D(ins),A(ins),
                                    L(ins),D(ins),gpr[D(ins)],A(ins),gpr[A(ins)]);
                                gpr[D(ins)]=gpr[A(ins)] >> L(ins);
                                break;
                            case 0x80: /* l.srai */
                                if(verbose&1)printf("l.srai  r%-2i, r%-2i, %7i    r%-2i=%08x r%-2i=%08x\n",
                                    D(ins),A(ins),
                                    L(ins),D(ins),gpr[D(ins)],A(ins),gpr[A(ins)]);
                                gpr[D(ins)]=(gpr[A(ins)] >> L(ins)) | (gpr[A(ins)]&0x80000000? POW2(L(ins)) :0);
                                break;
                            case 0x100: /* l.rori */
                                if(verbose&1)printf("l.rori   r%-2i, r%-2i, %7i    r%-2i=%08x r%-2i=%08x\n",
                                    D(ins),A(ins),
                                    L(ins),D(ins),gpr[D(ins)],A(ins),gpr[A(ins)]);
                                gpr[D(ins)] = (gpr[A(ins)] >> L(ins)) |
                                    (gpr[A(ins)] << (32-L(ins)));
                        }
                        break;
    
                    case 0x2f: /* l.sf*i */
                        switch(D(ins)) {
                            case 0: /* l.sfeqi */
                                if(verbose&1)printf("l.sfeqi  r%-2i, %7i    r%-2i=%08x\n",A(ins),
                                    I(ins),A(ins),gpr[A(ins)]);
                                flags=gpr[A(ins)] == I(ins);
                                break;
                            case 1: /* l.sfnei */
                                if(verbose&1)printf("l.sfnei  r%-2i, %7i    r%-2i=%08x\n",A(ins),
                                    I(ins),A(ins),gpr[A(ins)]);
                                flags=gpr[A(ins)] != I(ins);
                                break;
                            case 2: /* l.sfgtui */
                                if(verbose&1)printf("l.sfgtui r%-2i, %7i    r%-2i=%08x\n",A(ins),
                                    I(ins),A(ins),gpr[A(ins)]);
                                flags=gpr[A(ins)] > I(ins);
                                break;
                            case 3: /* l.sfgeui */
                                if(verbose&1)printf("l.sfgeui r%-2i, %7i    r%-2i=%08x\n",A(ins),
                                    I(ins),A(ins),gpr[A(ins)]);
                                flags=gpr[A(ins)] >= I(ins);
                                break;
                            case 4: /* l.sfltui */
                                if(verbose&1)printf("l.sfltui r%-2i, %7i    r%-2i=%08x\n",A(ins),
                                    I(ins),A(ins),gpr[A(ins)]);
                                flags=gpr[A(ins)] < I(ins);
                                break;
                            case 5: /* l.sfleui */
                                if(verbose&1)printf("l.sfleui r%-2i, %7i    r%-2i=%08x\n",A(ins),
                                    I(ins),A(ins),gpr[A(ins)]);
                                flags=gpr[A(ins)] <= I(ins);
                                break;
                            case 10: /* l.sfgtsi */
                                if(verbose&1)printf("l.sfgtsi r%-2i, %7i    r%-2i=%08x\n",A(ins),
                                    I(ins),A(ins),gpr[A(ins)]);
                                flags=(int)gpr[A(ins)] > I(ins);
                                break;
                            case 11: /* l.sfgesi */
                                if(verbose&1)printf("l.sfgesi r%-2i, %7i    r%-2i=%08x\n",A(ins),
                                    I(ins),A(ins),gpr[A(ins)]);
                                flags=(int)gpr[A(ins)] >= I(ins);
                                break;
                            case 12: /* l.sfltsi */
                                if(verbose&1)printf("l.sfltsi r%-2i, %7i    r%-2i=%08x\n",A(ins),
                                    I(ins),A(ins),gpr[A(ins)]);
                                flags=(int)gpr[A(ins)] < I(ins);
                                break;
                            case 13: /* l.sflesi */
                                if(verbose&1)printf("l.sflesi r%-2i, %7i    r%-2i=%08x\n",A(ins),
                                    I(ins),A(ins),gpr[A(ins)]);
                                flags=(int)gpr[A(ins)] <= I(ins);
                                break;
                        }
                        break;
                
                    case 0x35: /* l.sw */
                        if(verbose&1)printf("l.sw     %7i(r%-2i),r%-2i     r%-2i=%08x r%-2i=%08x\n",SI(ins),A(ins),
                    	        B(ins),A(ins),gpr[A(ins)],B(ins),gpr[B(ins)]);
                        SMEM(gpr[A(ins)] + SI(ins), gpr[B(ins)]);
                        if(forceex)exception(forceex,pc);
                        break;
                    case 0x36: /* l.sb */
                        if(verbose&1)printf("l.sb     %7i(r%-2i),r%-2i     r%-2i=%08x r%-2i=%08x\n",SI(ins),A(ins),
                    	        B(ins),A(ins),gpr[A(ins)],B(ins),gpr[B(ins)]);
                        SMEM8(gpr[A(ins)] + SI(ins), gpr[B(ins)]);
                        if(forceex)exception(forceex,pc);
                        break;
                    case 0x37: /* l.sh */
                        if(verbose&1)printf("l.sh     %7i(r%-2i),r%-2i     r%-2i=%08x r%-2i=%08x\n",SI(ins),A(ins),
                    	        B(ins),A(ins),gpr[A(ins)],B(ins),gpr[B(ins)]);
                        SMEM16(gpr[A(ins)] + SI(ins), gpr[B(ins)]);
                        if(forceex)exception(forceex,pc);
                        break;
                
            
                    case 0x38: /*  */
                        switch(ins&0xcf) {
                            case 0: /* l.add */
                                if(verbose&1)printf("l.add    r%-2i, r%-2i, r%-2i"
                                   "        r%-2i=%08x r%-2i=%08x r%-2i=%08x\n",
                                   D(ins),A(ins),B(ins),D(ins),gpr[D(ins)],A(ins),gpr[A(ins)],B(ins),gpr[B(ins)]);
                                gpr[D(ins)]=gpr[A(ins)] + gpr[B(ins)];
                                carry=gpr[D(ins)]<gpr[A(ins)];
                                break;
                            case 1: /* l.addc */
                                if(verbose&1)printf("l.addc   r%-2i, r%-2i, r%-2i"
                                   "        r%-2i=%08x r%-2i=%08x r%-2i=%08x carry=%i\n",
                                   D(ins),A(ins),B(ins),D(ins),gpr[D(ins)],
                                   A(ins),gpr[A(ins)],B(ins),gpr[B(ins)],carry);
                                gpr[D(ins)]=gpr[A(ins)] + gpr[B(ins)] + carry;
                                carry=(gpr[D(ins)]<gpr[A(ins)]) || (carry && !(carry+gpr[B(ins)]));
                                break;
                            case 2: /* l.sub */
                                if(verbose&1)printf("l.sub    r%-2i, r%-2i, r%-2i"
                                   "        r%-2i=%08x r%-2i=%08x r%-2i=%08x\n",
                                   D(ins),A(ins),B(ins),D(ins),gpr[D(ins)],A(ins),gpr[A(ins)],B(ins),gpr[B(ins)]);
                                gpr[D(ins)]=gpr[A(ins)] - gpr[B(ins)];
                                carry=gpr[D(ins)]>gpr[A(ins)];
                                break;
                            case 3: /* l.and */
                                if(verbose&1)printf("l.and    r%-2i, r%-2i, r%-2i"
                                   "        r%-2i=%08x r%-2i=%08x r%-2i=%08x\n",
                                   D(ins),A(ins),B(ins),D(ins),gpr[D(ins)],A(ins),gpr[A(ins)],B(ins),gpr[B(ins)]);
                                gpr[D(ins)]=gpr[A(ins)] & gpr[B(ins)];
                                break;
                            case 4: /* l.or */
                                if(verbose&1)printf("l.or     r%-2i, r%-2i, r%-2i"
                                   "        r%-2i=%08x r%-2i=%08x r%-2i=%08x\n",
                                   D(ins),A(ins),B(ins),D(ins),gpr[D(ins)],A(ins),gpr[A(ins)],B(ins),gpr[B(ins)]);
                                gpr[D(ins)]=gpr[A(ins)] | gpr[B(ins)];
                                break;
                            case 5: /* l.xor */
                                if(verbose&1)printf("l.xor    r%-2i, r%-2i, r%-2i"
                                   "        r%-2i=%08x r%-2i=%08x r%-2i=%08x\n",
                                   D(ins),A(ins),B(ins),D(ins),gpr[D(ins)],A(ins),gpr[A(ins)],B(ins),gpr[B(ins)]);
                                gpr[D(ins)]=gpr[A(ins)] ^ gpr[B(ins)];
                                break;
                            case 0xcb: /* l.mulu */
                                if(verbose&1)printf("l.mulu   r%-2i, r%-2i, r%-2i"
                                   "        r%-2i=%08x r%-2i=%08x r%-2i=%08x\n",
                                   D(ins),A(ins),B(ins),D(ins),gpr[D(ins)],A(ins),gpr[A(ins)],B(ins),gpr[B(ins)]);
                                gpr[D(ins)]=gpr[A(ins)] * gpr[B(ins)];
                                break;
                            case 0xc6: /* l.mul */
                                if(verbose&1)printf("l.mul    r%-2i, r%-2i, r%-2i"
                                   "        r%-2i=%08x r%-2i=%08x r%-2i=%08x\n",
                                   D(ins),A(ins),B(ins),D(ins),gpr[D(ins)],A(ins),gpr[A(ins)],B(ins),gpr[B(ins)]);
                                gpr[D(ins)]=(int)gpr[A(ins)] * (int)gpr[B(ins)];
                                break;
                            case 7: /* l.mac */
                                if(verbose&1)printf("l.mac    r%-2i, r%-2i, r%-2i"
                                   "        r%-2i=%08x r%-2i=%08x r%-2i=%08x\n",
                                   D(ins),A(ins),B(ins),D(ins),gpr[D(ins)],A(ins),gpr[A(ins)],B(ins),gpr[B(ins)]);
                                MAC += (int)gpr[A(ins)] * (int)gpr[B(ins)];
                                break;
                            case 0x08: /* shifts */
                                switch(ins&0xff) {
                                    case 0x38: /* l.ror */
                                        if(verbose&1)printf("l.ror    r%-2i, r%-2i, r%-2i"
                                             "        r%-2i=%08x r%-2i=%08x r%-2i=%08x\n",
                                             D(ins),A(ins),B(ins),D(ins),gpr[D(ins)],
                                             A(ins),gpr[A(ins)],B(ins),gpr[B(ins)]);
                                        gpr[D(ins)] = (gpr[A(ins)] >> gpr[B(ins)]) |
                                            (gpr[A(ins)] << (32-gpr[B(ins)]));
                                        break;
                                    case 0x08: /* l.sll */             
                                        if(verbose&1)printf("l.sll    r%-2i, r%-2i, r%-2i"
                                             "        r%-2i=%08x r%-2i=%08x r%-2i=%08x\n",
                                             D(ins),A(ins),B(ins),D(ins),gpr[D(ins)],
                                             A(ins),gpr[A(ins)],B(ins),gpr[B(ins)]);
                                        gpr[D(ins)]=gpr[A(ins)] << gpr[B(ins)];
                                        break;
                                    case 0x18: /* l.srl */
                                        if(verbose&1)printf("l.srl    r%-2i, r%-2i, r%-2i"
                                             "        r%-2i=%08x r%-2i=%08x r%-2i=%08x\n",
                                             D(ins),A(ins),B(ins),D(ins),gpr[D(ins)],
                                             A(ins),gpr[A(ins)],B(ins),gpr[B(ins)]);
                                        gpr[D(ins)]=gpr[A(ins)] >> gpr[B(ins)];
                                        break;
                                    case 0x28: /* l.sra */
                                        if(verbose&1)printf("l.sra    r%-2i, r%-2i, r%-2i"
                                             "        r%-2i=%08x r%-2i=%08x r%-2i=%08x\n",
                                             D(ins),A(ins),B(ins),D(ins),gpr[D(ins)],
                                             A(ins),gpr[A(ins)],B(ins),gpr[B(ins)]);
                                        gpr[D(ins)]=(gpr[A(ins)] >> gpr[B(ins)]) | 
                                            (gpr[A(ins)]&0x80000000? POW2(gpr[B(ins)]) :0);
                                        break;
                                }
                                break;
                            case 0x0c: /* l.exths */
                                gpr[D(ins)]= SEX16(gpr[A(ins)]);
                                break;
                            case 0x4c: /* l.extbs */
                                gpr[D(ins)]=SEX8(gpr[A(ins)]);
                                break;
                            case 0x8c: /* l.exthz */
                                gpr[D(ins)]= gpr[A(ins)] & 0xffff;
                                break;
                            case 0xcc: /* l.extbz */
                                gpr[D(ins)]= gpr[A(ins)] & 0xff;
                                break;
#if 1
                            case 9: /* l.div */
                                if(verbose&1)printf("l.div    r%-2i, r%-2i, r%-2i"
                                   "        r%-2i=%08x r%-2i=%08x r%-2i=%08x\n",
                                   D(ins),A(ins),B(ins),D(ins),gpr[D(ins)],A(ins),gpr[A(ins)],B(ins),gpr[B(ins)]);
                                if(gpr[B(ins)]) {
                                    gpr[D(ins)]=(int)gpr[A(ins)] / (int)gpr[B(ins)];
                                } else {
                                    /* XXX exception */
                                    overflow=1;
                                }
                                break;
                            case 0xa: /* l.divu */
                                if(verbose&1)printf("l.divu   r%-2i, r%-2i, r%-2i"
                                   "        r%-2i=%08x r%-2i=%08x r%-2i=%08x\n",
                                   D(ins),A(ins),B(ins),D(ins),gpr[D(ins)],A(ins),gpr[A(ins)],B(ins),gpr[B(ins)]);
                                if(gpr[B(ins)]) {
                                    gpr[D(ins)]=gpr[A(ins)] / gpr[B(ins)];
                                } else {
                                    /* XXX exception */
                                    overflow=1;
                                }
                                break;
#endif
                        }
                        break;
                
                    case 0x39: /* l.sf* */
                        switch(D(ins)) {
                            case 0: /* l.sfeq */
                                if(verbose&1)printf("l.sfeq   r%-2i, r%-2i"
                                     "             r%-2i=%08x r%-2i=%08x\n",
                                     A(ins),B(ins),A(ins),gpr[A(ins)],B(ins),gpr[B(ins)]);
                                flags=gpr[A(ins)] == gpr[B(ins)];
                                break;
                            case 1: /* l.sfne */
                                if(verbose&1)printf("l.sfne   r%-2i, r%-2i"
                                     "             r%-2i=%08x r%-2i=%08x\n",
                                     A(ins),B(ins),A(ins),gpr[A(ins)],B(ins),gpr[B(ins)]);
                                flags=gpr[A(ins)] != gpr[B(ins)];
                                break;
                            case 2: /* l.sfgtu */
                                if(verbose&1)printf("l.sfgtu  r%-2i, r%-2i"
                                     "             r%-2i=%08x r%-2i=%08x\n",
                                     A(ins),B(ins),A(ins),gpr[A(ins)],B(ins),gpr[B(ins)]);
                                flags=gpr[A(ins)] > gpr[B(ins)];
                                break;
                            case 3: /* l.sfgeu */
                                if(verbose&1)printf("l.sfgeu  r%-2i, r%-2i"
                                     "             r%-2i=%08x r%-2i=%08x\n",
                                     A(ins),B(ins),A(ins),gpr[A(ins)],B(ins),gpr[B(ins)]);
                                flags=gpr[A(ins)] >= gpr[B(ins)];
                                break;
                            case 4: /* l.sfltu */
                                if(verbose&1)printf("l.sfltu  r%-2i, r%-2i"
                                     "             r%-2i=%08x r%-2i=%08x\n",
                                     A(ins),B(ins),A(ins),gpr[A(ins)],B(ins),gpr[B(ins)]);
                                flags=gpr[A(ins)] < gpr[B(ins)];
                                break;
                            case 5: /* l.sfleu */
                                if(verbose&1)printf("l.sfleu  r%-2i, r%-2i"
                                     "             r%-2i=%08x r%-2i=%08x\n",
                                     A(ins),B(ins),A(ins),gpr[A(ins)],B(ins),gpr[B(ins)]);
                                flags=gpr[A(ins)] <= gpr[B(ins)];
                                break;
                            case 10: /* l.sfgts */
                                if(verbose&1)printf("l.sfgts  r%-2i, r%-2i"
                                     "             r%-2i=%08x r%-2i=%08x\n",
                                     A(ins),B(ins),A(ins),gpr[A(ins)],B(ins),gpr[B(ins)]);
                                flags=(int)gpr[A(ins)] > (int)gpr[B(ins)];
                                break;
                            case 11: /* l.sfges */
                                if(verbose&1)printf("l.sfges  r%-2i, r%-2i"
                                     "             r%-2i=%08x r%-2i=%08x\n",
                                     A(ins),B(ins),A(ins),gpr[A(ins)],B(ins),gpr[B(ins)]);
                                flags=(int)gpr[A(ins)] >= (int)gpr[B(ins)];
                                break;
                            case 12: /* l.sflts */
                                if(verbose&1)printf("l.sflts  r%-2i, r%-2i"
                                     "             r%-2i=%08x r%-2i=%08x\n",
                                     A(ins),B(ins),A(ins),gpr[A(ins)],B(ins),gpr[B(ins)]);
                                flags=(int)gpr[A(ins)] < (int)gpr[B(ins)];
                                break;
                            case 13: /* l.sfles */
                                if(verbose&1)printf("l.sfles  r%-2i, r%-2i"
                                     "             r%-2i=%08x r%-2i=%08x\n",
                                     A(ins),B(ins),A(ins),gpr[A(ins)],B(ins),gpr[B(ins)]);
                                flags=(int)gpr[A(ins)] <= (int)gpr[B(ins)];
                                break;
                        }
                        break;
            
                    default:
                        exception(EX_ILL, 0);
                        break;
                
                }
            }
        }
        if(++nextjump) {
            pc+=4;
        } else {
            pc=newpc;
        }
        
    }
    return 0;
}
