#include <linux/config.h>
#undef CONFIG_MODVERSIONS
#include <linux/module.h>
#include <linux/version.h>
#include <linux/modversions.h>

#include <linux/init.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/slab.h>
#include <linux/pci.h>

#include <asm/uaccess.h>
#include <asm/io.h>

#include <asm/system.h>   /* cli(), *_flags */
#include <asm/segment.h>  /* memcpy and such */
#include <asm/pgtable.h>

#include "bcm.h"

#define DEVNAME "BCM"

static struct bcm_dev *devices[MAXDEVICES];
static int num_devices;

static int tx_size=16, rx_size=16;
MODULE_PARM(tx_size, "i");
MODULE_PARM(rx_size, "i");

static int verbose=1;
MODULE_PARM(verbose, "i");

static int rx_bufsize=4096;
MODULE_PARM(rx_bufsize, "i");

static struct protocol *find_protocol(struct bcm_dev *bdev, u32 proto);

static void add_rx_element(struct bcm_dev *bdev) {
	int head, tail;
	head=__le32_to_cpu(bdev->common->rx_head);
	tail=bdev->rx_host_tail;
	if((head+1==tail) || (head==bdev->rx_size-1 && tail==0)) return;
	bdev->rx_ring[head].bufsize=__cpu_to_le32(rx_size);
	bdev->rx_ring[head].flags=__cpu_to_le32(FLAGS_OWNER_CARD);
	bdev->rx_ring[head].buffer=kmalloc(rx_bufsize, GFP_KERNEL);
	bdev->rx_ring[head].phys=__cpu_to_le64(virt_to_phys(bdev->rx_ring[head].buffer));
	head++;
	if(head==bdev->rx_size-1)head=0;
	bdev->common->rx_head=__cpu_to_le32(head);
}

static void rx_task(void *data) {
	struct bcm_dev *bdev=data;
	int head, tail;
	head=__le32_to_cpu(bdev->common->rx_tail); /* the head of our list is the tail of the card */
	tail=bdev->rx_host_tail;
	while(head!=tail) {
		struct protocol *p;
		p=find_protocol(bdev, __le32_to_cpu(bdev->rx_ring[tail].proto));
		if(p) {
			struct bcm_transfer *tr;
			tr=kmalloc(sizeof(struct bcm_transfer), GFP_ATOMIC);
			tr->dev=bdev;
			tr->buffer=bdev->rx_ring[tail].buffer;
			tr->len=__le32_to_cpu(bdev->rx_ring[tail].len);
			tr->proto=__le32_to_cpu(bdev->rx_ring[tail].proto);
			p->payload->rx_action(bdev, tr);
		} else {
			kfree(bdev->rx_ring[tail].buffer);
		}
		tail++;
		if(tail==bdev->rx_size)tail=0;
		bdev->rx_host_tail=tail;
		add_rx_element(bdev);
		head=__le32_to_cpu(bdev->common->rx_tail);	
	}
}

static void tx_task(void *data) {
	struct bcm_dev *bdev=data;
	struct bcm_transfer * tr;

	tr=bdev->tx_finished;
	bdev->tx_finished=NULL;
	while(tr!=NULL){
		struct protocol *p;
		struct bcm_transfer *tmp;
		tmp=tr->next;
		if(/*tr->flags&FLAGS_NOTIFY &&*/ (p=find_protocol(bdev,tr->proto))) {
			p->payload->tx_notify(bdev, tr);
		} else {
			kfree(tr->buffer);
			kfree(tr);
		}
		tr=tmp;
	}
}		

static void irq_handler(int irq, void *dev_id, struct pt_regs *regs) {
	int msg, head, tail;
	struct bcm_dev *bdev = (struct bcm_dev *)dev_id;
	
	msg=__le32_to_cpu(bdev->common->msg);
	if(verbose)printk(KERN_DEBUG DEVNAME ": interrupt, message is %i\n",msg);
	switch(msg) {
		case 0: /* this should not happen */
			break;
		case 1: /* ready to roll */
			break;
		case 2: /* card wrote to rx ring */
			schedule_task(&bdev->tq_rx);
			break;
		case 3: /* card notifies of a finished tx */
			head=__le32_to_cpu(bdev->common->tx_tail);
			tail=bdev->tx_host_tail;
			while(head!=tail) {
				struct bcm_transfer *tr;

				tr=kmalloc(sizeof(struct bcm_transfer), GFP_ATOMIC);
				tr->buffer=bdev->tx_ring[tail].buffer;
				tr->len=__le32_to_cpu(bdev->tx_ring[tail].len);
				tr->proto=__le32_to_cpu(bdev->tx_ring[tail].proto);
				tr->flags=__le32_to_cpu(bdev->tx_ring[tail].flags);
				tr->next=bdev->tx_finished;
				bdev->tx_finished=tr;
				tail+=1;
				if(tail==bdev->tx_size)tail=0;
				bdev->tx_host_tail=tail;
				head=__le32_to_cpu(bdev->common->tx_tail);
			}
			schedule_task(&bdev->tq_tx);
			break;
		default:
			if(verbose)printk(KERN_WARNING DEVNAME ": Unexpected interrupt %i\n",msg);
	}
	bdev->common->msg=0;
	bdev->mbox0[6]=0x400; /* acknowledge interrupt EW */	
}

static int __init s(void) {
	struct pci_dev * dev;
	
	num_devices=0;
	dev = pci_find_device(0x166d, 0x0001, NULL);
	while (dev && num_devices<MAXDEVICES) {
		struct bcm_dev *bdev;
		int i;
		
		devices[num_devices]=kmalloc(sizeof(struct bcm_dev), GFP_KERNEL);
		bdev=devices[num_devices];
		bdev->dev=dev;
		
		pci_enable_device(dev);
		pci_set_master(dev);

		bdev->common=ioremap(pci_resource_start(dev,0), 0x1000);
		bdev->mbox0=ioremap(pci_resource_start(dev,2), 0x1000);

		if(verbose) printk(KERN_NOTICE DEVNAME": Found board, id  %i %i %i %i\n",
				bdev->common->signature[0], 
				bdev->common->signature[1], 
				bdev->common->signature[2], 
				bdev->common->signature[3]);
		
		request_irq(dev->irq, irq_handler, SA_SHIRQ, DEVNAME, bdev);

		bdev->tx_size=tx_size;
		bdev->rx_size=rx_size;
				
		bdev->tx_ring=kmalloc(bdev->tx_size*sizeof(ring_element), GFP_KERNEL);
		for(i=0;i<bdev->tx_size;i++)bdev->tx_ring[i].flags=FLAGS_OWNER_HOST;
		bdev->rx_ring=kmalloc(bdev->rx_size*sizeof(ring_element), GFP_KERNEL);

		bdev->common->tx_size=__cpu_to_le32(bdev->tx_size);
		bdev->common->tx_phys=__cpu_to_le64(virt_to_phys(bdev->tx_ring));
		bdev->common->rx_size=__cpu_to_le32(bdev->rx_size);
		bdev->common->rx_phys=__cpu_to_le64(virt_to_phys(bdev->rx_ring));
		
		bdev->common->rx_head=0;
		bdev->common->rx_tail=0;
		bdev->rx_host_tail=0;
		bdev->tx_host_tail=0;
		bdev->common->tx_head=0;
		bdev->common->tx_tail=0;
		
		for(i=2;i<bdev->rx_size;i++)add_rx_element(bdev);

		bdev->tq_rx.data=bdev;
		bdev->tq_rx.routine=rx_task;
		
		bdev->tq_tx.data=bdev;
		bdev->tq_tx.routine=tx_task;
		
		bdev->mbox0[6]=0x100; /* Signal card to read data EW */
		num_devices++;

		bdev->prot=NULL;
		bdev->tx_finished=NULL;
		dev = pci_find_device(0x166d, 0x0001, dev);
	}

	return 0;
}

static void __exit e(void) {
	int i;
	for(i=0;i<num_devices;i++) {
		free_irq(devices[i]->dev->irq, devices[i]);
		iounmap(devices[i]->common);
		iounmap(devices[i]->mbox0);
		kfree(devices[i]->tx_ring);
		kfree(devices[i]->rx_ring);
		kfree(devices[i]);
		pci_disable_device(devices[i]->dev);
	}
	return ;
}

module_init(s);
module_exit(e);


static struct protocol *find_protocol(struct bcm_dev *bdev, u32 proto) {
	struct protocol *p;
	
	p=bdev->prot;
	while(p!=NULL && p->proto!=proto) p=p->next;
	return p;
}

/*
 * external interface
 */

int bcm_register_driver(struct bcm_driver *dr) {
	int i;
	struct dev_info di;

	for(i=0;i<num_devices;i++) {
		di.bcm=devices[i];
		strcpy(di.identity, "BCM");
		dr->init(&di);
	}
	return 0;
}

int bcm_register_payload(struct bcm_dev *bdev, struct bcm_payload *pl) {
	struct protocol *p;
	
	p=find_protocol(bdev, pl->proto);
	if(p!=NULL) {
		p->payload=pl;
	} else {
		p=bdev->prot;
		bdev->prot=kmalloc(sizeof(struct protocol), GFP_KERNEL);
		bdev->prot->next=p;
		bdev->prot->payload=pl;
		bdev->prot->proto=pl->proto;
	}
	return 0;
}

int bcm_unregister_payload(struct bcm_dev *bdev, struct bcm_payload *pl) {
	struct protocol *p;
	p=bdev->prot;
	while(p!=NULL && p->payload!=pl)p=p->next;
	if(p!=NULL) {
		if(p==bdev->prot) {
			bdev->prot=p->next;
			kfree(p);
		} else {
			struct protocol *t;
			t=bdev->prot;
			while(t!=NULL && t->next!=p)t=t->next;
			if(t->next==p) {
				t->next=p->next;
				kfree(p);
			}
		}
		return 0;
	}
	return -1;
}

int bcm_transfer_data(struct bcm_transfer *tr) {
	int head, tail;
	struct bcm_dev *bdev;
	u32 flags;

	bdev=tr->dev;
	head=__le32_to_cpu(bdev->common->tx_head);
	tail=__le32_to_cpu(bdev->common->tx_tail);
	if(head==tail-1 || (head==bdev->tx_size-1 && tail==0)) return -1;
	bdev->tx_ring[head].phys=__cpu_to_le64(virt_to_phys(tr->buffer));
	bdev->tx_ring[head].buffer=tr->buffer;
	bdev->tx_ring[head].len=__cpu_to_le32(tr->len);
	bdev->tx_ring[head].proto=__cpu_to_le32(tr->proto);
	flags=FLAGS_OWNER_CARD;
	if(find_protocol(bdev, tr->proto)!=NULL) flags |= FLAGS_NOTIFY;
	bdev->tx_ring[head].flags=__cpu_to_le32(flags);
	head++;
	if(head==bdev->tx_size)head=0;
	bdev->common->tx_head=__cpu_to_le32(head);
	bdev->mbox0[6]=0x200;
	return 0;
}

EXPORT_SYMBOL(bcm_register_driver);
EXPORT_SYMBOL(bcm_register_payload);
EXPORT_SYMBOL(bcm_unregister_payload);
EXPORT_SYMBOL(bcm_transfer_data);
MODULE_LICENSE("GPL");
