#include <vga.h>
#include <vgakeyboard.h>
#include <stdio.h>
#include <signal.h>

#include "bd.h"
#include "keys.h"

int color_fg;
int color_bg;
int color_uitext;
int color_selected;
int color_cursor;
int color_decor;

#define KEYBUFSIZE 256

static char zero[8192];
volatile int keybuffer[KEYBUFSIZE];
volatile int shiftbuffer[KEYBUFSIZE];
volatile int firstkey, lastkey;
static int shift_state; /* shift capslock ctrl alt numlock */
static int shiftkeys; /* rshift lshift capslock rctrl lctrl ralt lalt numlock */

void handler(int scancode, int newstate)
{
    int m;

    switch(scancode) {
        case SCANCODE_RIGHTSHIFT:
            m=1;
            break;
        case SCANCODE_LEFTSHIFT:
            m=2;
            break;
        case SCANCODE_CAPSLOCK:
            m=4;
            break;
        case SCANCODE_RIGHTCONTROL:
            m=8;
            break;
        case SCANCODE_LEFTCONTROL:
            m=16;
            break;
        case SCANCODE_RIGHTALT:
            m=32;
            break;
        case SCANCODE_LEFTALT:
            m=64;
            break;
        case SCANCODE_NUMLOCK:
            m=128;
            break;
        default:
            m=0;
    }
    if(m) {
        if(newstate)shiftkeys|=m; else shiftkeys&=~m;
        switch(scancode) {
            case SCANCODE_CAPSLOCK:
                if(newstate)shift_state^=2;
                break;
            case SCANCODE_NUMLOCK:
                if(newstate)shift_state^=16;
                break;
            case SCANCODE_RIGHTSHIFT: case SCANCODE_LEFTSHIFT:
                shift_state&=0xfe;
                shift_state |= shiftkeys&3 ? 1 : 0;
                break;
            case SCANCODE_RIGHTCONTROL: case SCANCODE_LEFTCONTROL:
                shift_state&=0xfb;
                shift_state |= shiftkeys&24 ? 4 : 0;
                break;
            case SCANCODE_RIGHTALT: case SCANCODE_LEFTALT:
                shift_state&=0xf7;
                shift_state |= shiftkeys&96 ? 8 : 0;
                break;
        }
    } else {
        if(newstate) {
            if(((firstkey+1)%KEYBUFSIZE)==lastkey) {
                printf("Buffer full\n");
            } else {
                shiftbuffer[firstkey]=shift_state;
                keybuffer[firstkey++]=scancode;
                if(firstkey==KEYBUFSIZE)firstkey=0;
            }
        }
    }
}

int readchar() {

    int ch;
    
    ch=0;
    while(ch==0) {
        vga_waitevent(VGA_KEYEVENT|VGA_MOUSEEVENT, NULL, NULL, NULL, NULL);
        if(firstkey!=lastkey) {
            ch=translate_keycode(keybuffer[lastkey], shiftbuffer[lastkey]);
            lastkey++;
            if(lastkey==KEYBUFSIZE)lastkey=0;
        }
    }
    
    return ch;
}

int graphics_changesize() {
    int vgamode;

    switch (display_width*10000+display_pix_height) {
        case  3200200:
            vgamode=G320x200x256;
            break;
        case  3200240:
            vgamode=G320x240x256V;
            break;
        case  6400480:
            vgamode=G640x480x256;
            break;
        case  8000600:
            vgamode=G800x600x256;
            break;
        case 10240768:
            vgamode=G1024x768x256;
            break;
        case 11520864:
            vgamode=G1152x864x256;
            break;
        case 12801024:
            vgamode=G1280x1024x256;
            break;
        case 16001200:
            vgamode=G1600x1200x256;
            break;
        case 20481536:
            vgamode=G2048x1536x256;
            break;
        default:
            exit(1);
    }

    vga_setmode(vgamode);
    vga_setlinearaddressing();
    graphbase=vga_getgraphmem();

    vga_setmousesupport(1);

    return 0;
}

int init_graphics() {
    int vgamode;

    switch (display_width*10000+display_pix_height) {
        case  3200200:
            vgamode=G320x200x256;
            break;
        case  3200240:
            vgamode=G320x240x256V;
            break;
        case  6400480:
            vgamode=G640x480x256;
            break;
        case  8000600:
            vgamode=G800x600x256;
            break;
        case 10240768:
            vgamode=G1024x768x256;
            break;
        case 11520864:
            vgamode=G1152x864x256;
            break;
        case 12801024:
            vgamode=G1280x1024x256;
            break;
        case 16001200:
            vgamode=G1600x1200x256;
            break;
        case 20481536:
            vgamode=G2048x1536x256;
            break;
        default:
            exit(1);
    }

    vga_disabledriverreport();
    vga_init();
    vga_setmode(vgamode);
    vga_setlinearaddressing();
    screen_bpp=1;

    vis_pitch=vis_width*screen_bpp;
    screen_pitch=display_width*screen_bpp;

    graphbase=vga_getgraphmem();

    color_bg=15;
    color_fg=0;
    color_uitext=7;
    color_selected=4;
    color_cursor=0x87;
    color_decor=13;

    memset(zero,color_bg,8192);
    
    firstkey=0;
    lastkey=0;
    keyboard_init();
    shift_state=0;
    keyboard_seteventhandler(handler);
    signal(SIGIO,SIG_IGN);
    signal(SIGPOLL,SIG_IGN);
    return 0;
}

int close_graphics() {
    keyboard_close();
    vga_setmode(TEXT);
    return 0;
}

int save_area(int x, int y, int pitch, int height, char *buf)
{
    int i;
    
    for(i=0;i<height;i++) {
        vga_getscansegment(buf+pitch*i, x, y+i, pitch);
    }
    return 0;
}

int restore_area(int x, int y, int pitch, int height, char *buf)
{
    int i;
    
    for(i=0;i<height;i++) {
        vga_drawscansegment(buf+pitch*i, x, y+i, pitch);
    }
    return 0;
}

int blank_area(int x, int y, int pitch, int height)
{
    int i;
    
    for(i=0;i<height;i++) {
        vga_drawscansegment(zero, x, y+i, pitch);
    }
    return 0;
}

int drawline(int x1, int y1, int x2, int y2, int col) {
/* only vertical or horizontal */
    char *m;
    int d,n;
    
    m=graphbase+screen_pitch*y1+x1*screen_bpp;
    if(x2>x1) {
        d=screen_bpp;
        n=x2-x1;
    } else if (y1<y2) {
        d=screen_pitch;   
        n=y2-y1;
    } else return -1;

    for(;n>0;n--) {
        switch(screen_bpp) {
            case 1:
                *m=col;
                break;
            case 2:
                *(short *)m=col;
                break;
            case 4:
                *(long *)m=col;
                break;
            case 3:
                *m=col&0xff;
                *(m+1)=(col>>16)&0xff;
                *(m+2)=(col>>24)&0xff;
        }
        m+=d;
    }
                
    return 0;
}

