#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/time.h>

#include <string.h>
#include <SDL.h>
#include "const.h"

#define INRECT(px,py, x,y,w,h) ((px)>(x) && (px)<(x)+(w) && (py)>(y) && (py)<(y)+(h))

void drawkeyboard(struct shared_t *emudata, int kdisp);
int xytochar(int x, int y);
void loadsnap(struct shared_t *emudata, char *name, struct prefs_t *prefs);
void savesnap(struct shared_t *emudata, char *name, struct prefs_t *prefs);
void run_audio(struct status_t *stat, struct shared_t *emudata);
void uninit_audio(void);
void pause_audio(struct status_t *stat);
void unpause_audio(struct status_t *stat);
void setkey(int c, struct shared_t *emudata);
void resetkey(int c, struct shared_t *emudata);
int getkey(int s);
int drawchar2(struct shared_t *emudata, int x, int y, int c, int col);
int handlekeys(struct shared_t *emudata, int key, char *name);
void selectfile(struct shared_t *emudata);

static unsigned char memrom[16384];
static unsigned char memram[49152];

struct ui_e uis [7]= { 
	{0,  0,50,50,COL2,"RESET"},
	{60, 0,50,50,COL2,"SOUND"},
	{120,0,50,50,COL2,"FAST"},
	{180,0,50,50,COL2,"QUIT"},
	{0,80,50,50,COL1,"PAUSE"},
	{60,80,50,50,COL1,"LOAD"},
//	{120,80,50,50,COL1,"QUIT"}, 
//	{180,80,50,50,COL1,"QUIT"},
	{-1}
};

void refresh(struct shared_t *);
void mainloop(struct shared_t *);

extern struct hkeys_s hkeys[];

void set_default_prefs(struct prefs_t *prefs) {
	prefs->kdisp=1;
	prefs->keymode=1;
	prefs->fast=0;
	prefs->sound=1;
	memcpy(prefs->hardkeys, hkeys[0].keys, 11);
}

int main(int argc, char *argv[]) {
	int done;
	int f, n, i;
    UInt8 lastkey[256];
    int lastshift=0;
	int lastpen;
	int keystotal;
	struct prefs_t localp;
	struct timeval starttime, ptime;
	struct status_t stat;
	SDL_Surface *screen;
	unsigned short *sc;
	int pcounter;

	struct shared_t emudata;

	putenv("SDL_VIDEO_X11_WMCLASS=maemus");

	set_default_prefs(&localp);

	f=open("spectrum.rom", O_RDONLY);
	if(f==-1) f=open("/usr/share/maemus/roms/spectrum.rom", O_RDONLY);
	if(f==-1) {
		fprintf(stderr,"No rom, no emu.\n");
		exit(1);
	}
	read(f, memrom, 16384);
	close(f);
	
	/* Initialize SDL */
	SDL_Init(SDL_INIT_VIDEO|SDL_INIT_AUDIO);

	atexit(SDL_Quit);
	SDL_WM_SetCaption("Maemus", NULL);

	/* Initialize the screen / window */
	screen = SDL_SetVideoMode(SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_DEPTH,
			SDL_HWSURFACE | SDL_FULLSCREEN);
	SDL_LockSurface(screen);
#ifdef __arm__
	SDL_ShowCursor(SDL_DISABLE);
#endif
		
   	sc = (unsigned short *)screen->pixels;
	
	emudata.screen=screen;
	emudata.sc=sc;
	emudata.ram=memram;
	emudata.rom=memrom;
	
	drawkeyboard( &emudata, 1);
     
	for(n = 0; n < 256; n++) {
		lastkey[n] = 0;
	}
	keystotal=0;

	drawchar2( &emudata, U_X+uis[1].x+20, U_Y+uis[1].y+30, localp.sound+0x30, 0);
	drawchar2( &emudata, U_X+uis[2].x+20, U_Y+uis[2].y+30, localp.fast+0x30, 0);
	SDL_Flip(screen);

	stat.fast=localp.fast;
	stat.sndinited=0;
	stat.sndwant=0;
	stat.sndrunning=0;
	stat.pause=0;
	stat.mode=0;

	done=0;
	lastpen=0;

	for(i=0;i<8;i++)emudata.k[i]=0xff;
	emudata.k[8]=0x1f;
	emudata.out254=255;
	emudata.lastborder=0;
	emudata.lastsound=16;
	emudata.r=0x80;
	emudata.iff1=0;
	emudata.iff2=0;
	emudata.im=1;
	emudata.pc=0;
	emudata.counter=0;
	emudata.scounter=0;
	emudata.tcounter=0;

	if(argc>1) {
		loadsnap( &emudata, argv[1], &localp);
	}

	gettimeofday( &starttime, NULL);
	ptime=starttime;
	pcounter=emudata.counter;

	if(localp.sound && !stat.fast) {
		run_audio( &stat, &emudata);
	}

	while (!done) {
		struct timeval tv;
		SDL_Event event;
		int x,y;
		SDL_Rect r;
		
		if(stat.pause) {
			SDL_WaitEvent(NULL);
		} else {
			gettimeofday( &tv, NULL);

			if(1000000*(tv.tv_sec-ptime.tv_sec)+tv.tv_usec-ptime.tv_usec>=2000000) {
				char t[10];
				int l;
				
				r.x=uis[2].x+U_X;
				r.y=uis[2].y+uis[2].h+2+U_Y;
				r.h=10;
				r.w=uis[2].w;
				SDL_FillRect(screen, &r, 0);
				ptime=tv;
				l=sprintf(t, "%i", emudata.counter-pcounter);
				for(l-- ; l>=0 ; l--) 
					drawchar2( &emudata, U_X+uis[2].x+8*l, 
							U_Y+uis[2].y+uis[2].h+2, t[l], COL2);

				SDL_UpdateRect(screen, U_X+uis[2].x, U_Y+uis[2].y+uis[2].h+2, uis[2].w, 10);
				pcounter=emudata.counter;
			}

			if(stat.fast || 
			   emudata.counter-emudata.scounter<2 ||  
			   (20000*(emudata.counter-emudata.tcounter-
					   50*(tv.tv_sec-starttime.tv_sec))) < 
					tv.tv_usec-starttime.tv_usec)
			{
				if(keystotal){
					keystotal=0;
					for(n = 0; n < 256; n++) {
						if(lastkey[n]) {
							lastkey[n]--;
							if(!lastkey[n]) {
								resetkey(n, &emudata);
							} else keystotal=1;
						}
					}
				}

				emudata.tstates = 0;
				emudata.event_next_event = 69888;

//gettimeofday( &tv, NULL); fprintf(stderr, " %6i ", tv.tv_usec);
				mainloop(&emudata);
//gettimeofday( &tv, NULL); fprintf(stderr, " %6i ", tv.tv_usec);
				if(emudata.counter-emudata.scounter>=1 
						&& (stat.fast!=2 || !(emudata.counter&3))
						) {
					refresh(&emudata);
//gettimeofday( &tv, NULL); fprintf(stderr, " %6i ", tv.tv_usec);
					SDL_UpdateRect(screen, S_X, S_Y, 512, 384);
				};
//gettimeofday( &tv, NULL); fprintf(stderr, " %6i\n", tv.tv_usec);

				emudata.counter++;
				emudata.lastborder=emudata.out254;
				
				if(stat.sndwant && 
						emudata.counter-emudata.scounter>2) {
					unpause_audio(&stat);
				}
				
				if(++emudata.sndwrite == NUMSNDBUFS)emudata.sndwrite=0;
			}
		}
		
		if (SDL_PollEvent(&event)) switch (event.type) {
			char fname[256];
			case SDL_ACTIVEEVENT:
				if(!event.active.gain) {
					stat.pause=1;
					uninit_audio();
				} else {
					stat.pause=0;
					gettimeofday( &starttime, NULL);
					emudata.tcounter=emudata.counter;
					if(localp.sound) {
						run_audio(&stat, &emudata);
					}
				}
				break;
			case SDL_QUIT:
				done = 1;
				break;
			case SDL_KEYDOWN:
				switch(stat.mode) {
					case 0:
						if(getkey(event.key.keysym.sym)>=0)
							setkey(localp.hardkeys[getkey( (event.key.keysym.sym))], &emudata);
							else done=1;
						break;
					case 1:
						switch(event.key.keysym.sym) {
							case SDLK_LEFT:
								x='5';
								break;
							case SDLK_DOWN:
								x='6';
								break;
							case SDLK_UP:
								x='7';
								break;
							case SDLK_RIGHT:
								x='8';
								break;
							case SDLK_RETURN:
								x='0';
								break;
							default:
								x=0;
						}
						x=handlekeys(&emudata, x, fname);
						if(x==1) {
							if(fname[0])
								loadsnap(&emudata, fname, &localp);
							stat.mode=0;
							stat.pause=0;
							gettimeofday( &starttime, NULL);
							emudata.tcounter=emudata.counter;
							if(localp.sound) {
								run_audio(&stat, &emudata);
							}
						}
						
						break;
				}
				break;
			case SDL_KEYUP:
				if(getkey(event.key.keysym.sym)>=0)
					resetkey(localp.hardkeys[getkey( (event.key.keysym.sym))], &emudata);
				break;
            case SDL_MOUSEBUTTONDOWN:
				x=event.button.x;
				y=event.button.y;
				if (stat.mode==1 && INRECT(x,y,S_X+32,S_Y+48, 448, 288)) {
					int k;
					k= x> S_X+256;
					k+=((y-S_Y-48)/16)*2;
					x=handlekeys(&emudata, k+64, fname);
				}	
				if (x>K_X && x<K_X+K_W && y>K_Y && y<K_Y+K_H) {
					int c;
					c= xytochar(x-K_X,y-K_Y);
 					if(c>=0) {
 						lastkey[c] = 255;
						setkey(c, &emudata);
						lastpen=c;
						keystotal=1;
					}
				}
				if(x>U_X && x<U_X+U_W && y>U_Y && y< U_Y+U_W) {
					int i;
					i=0;
					while(1) {
						if (uis[i].x==-1) break;
						if ( INRECT(x-U_X,y-U_Y, uis[i].x, uis[i].y, uis[i].w, uis[i].h)) 
							break;
						i++;
					}
					if (uis[i].x>=0) {
						switch(i) {
							case 0:
								emudata.pc=0;
								emudata.r=0x80;
								emudata.iff1=0;
								emudata.iff2=0;
								emudata.im=1;
								/* reset */
								break;
							case 1:
								/* sound */
								localp.sound=!localp.sound;
								r.x=uis[1].x+20+U_X;
								r.y=uis[1].y+30+U_Y;
								r.h=10;
								r.w=10;
								SDL_FillRect(screen, &r, uis[1].c);
								drawchar2( &emudata, U_X+uis[1].x+20, U_Y+uis[1].y+30, 
										localp.sound+0x30, 0);
								SDL_UpdateRect(screen, r.x, r.y, r.h, r.w);
								if(localp.sound) {
									stat.fast=0;
									r.x=uis[2].x+20+U_X;
									r.y=uis[2].y+30+U_Y;
									r.h=10;
									r.w=10;
									SDL_FillRect(screen, &r, uis[2].c);
									drawchar2( &emudata, U_X+uis[2].x+20, U_Y+uis[2].y+30, 
											stat.fast+0x30, 0);
									SDL_UpdateRect(screen, r.x, r.y, r.h, r.w);
									run_audio(&stat, &emudata);
								} else {
									pause_audio(&stat);
								}
								break;
							case 2:
								if(++stat.fast==3) stat.fast=0;
								r.x=uis[2].x+20+U_X;
								r.y=uis[2].y+30+U_Y;
								r.h=10;
								r.w=10;
								SDL_FillRect(screen, &r, uis[2].c);
								drawchar2( &emudata, U_X+uis[2].x+20, U_Y+uis[2].y+30, 
										stat.fast+0x30, 0);
								SDL_UpdateRect(screen, r.x, r.y, r.h, r.w);
								if(!stat.fast) {
									gettimeofday( &starttime, NULL);
									emudata.tcounter=emudata.counter;
									if(localp.sound) {
										run_audio(&stat, &emudata);
									}
								} else {
									pause_audio(&stat);
								}
								break;
							case 3:
								done=1;
								break;
							case 4:
								stat.pause=!stat.pause;
								if(stat.pause) {
									pause_audio(&stat);
								} else {
									gettimeofday( &starttime, NULL);
									emudata.tcounter=emudata.counter;
									if(localp.sound) {
										run_audio(&stat, &emudata);
									}
								}
								break;
							case 5:
								stat.mode=!stat.mode;
								if(stat.mode) {
									stat.pause=1;
									pause_audio(&stat);
									selectfile( &emudata);
								} else {
									stat.pause=0;
									gettimeofday( &starttime, NULL);
									emudata.tcounter=emudata.counter;
									if(localp.sound) {
										run_audio(&stat, &emudata);
									}
								}
						}
					}
				}
                break;
            case SDL_MOUSEBUTTONUP:
				if(lastpen) {
					if(localp.keymode && (lastpen==22 || lastpen==24)) {
						lastshift=lastpen;
					} else {
						lastkey[lastpen]=1;
						if(lastshift) {
							lastkey[lastshift]=1;
							lastshift=0;
						}
					}
					lastpen=0;
				}
                break;
			default:
				break;
		}
		if(!stat.fast && emudata.counter >= emudata.scounter)usleep(1);
	}
printf("counter=%i\n scounter=%i\n sndread=%i\n sndwrite=%i\n",
		emudata.counter, emudata.scounter, emudata.sndwrite, emudata.sndread);
	uninit_audio();
	savesnap( &emudata, "tmp.sna", &localp);
	return 0;
}
