
/* dbus-monitor.c  Utility program to monitor messages on the bus
 *
 * Copyright (C) 2003 Philip Blundell <philb@gnu.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <glib.h>
#include <dbus/dbus-glib-lowlevel.h>
#include <dbus/dbus.h>

char *msgs[16];
char text[2048];
int nmsgs;

static const char*
type_to_name (int message_type)
{
  switch (message_type)
    {
    case DBUS_MESSAGE_TYPE_SIGNAL:
      return "signal";
    case DBUS_MESSAGE_TYPE_METHOD_CALL:
      return "method_call";
    case DBUS_MESSAGE_TYPE_METHOD_RETURN:
      return "method_return";
    case DBUS_MESSAGE_TYPE_ERROR:
      return "error";
    default:
      return "unknown";
    }
}

static void
print_iter (DBusMessageIter *iter, dbus_bool_t literal, int depth)
{
  do {
      int type = dbus_message_iter_get_arg_type (iter);
      const char *str;
      dbus_uint32_t uint32;
      dbus_int32_t int32;
      double d;
      unsigned char byte;
      dbus_bool_t boolean;

      if (type == DBUS_TYPE_INVALID) break;

      switch (type) {
		case DBUS_TYPE_STRING:
          dbus_message_iter_get_basic (iter, &str);
		  strncpy(msgs[nmsgs], str,127);
		  nmsgs++;
	  	  break;
#if 0
		case DBUS_TYPE_INT32:
          dbus_message_iter_get_basic (iter, &int32);
	  	  printf ("int32 %d\n", int32);
	  	  break;

		case DBUS_TYPE_UINT32:
          dbus_message_iter_get_basic (iter, &uint32);
	  	  printf ("uint32 %u\n", uint32);
	  	  break;
#endif

#if 0
		case DBUS_TYPE_DOUBLE:
	  		dbus_message_iter_get_basic (iter, &d);
	  		printf ("double %g\n", d);
	  		break;

		case DBUS_TYPE_BYTE:
			dbus_message_iter_get_basic (iter, &byte);
			printf ("byte %d\n", byte);
		    break;

		case DBUS_TYPE_BOOLEAN:
  			dbus_message_iter_get_basic (iter, &boolean);
			printf ("boolean %s\n", boolean ? "true" : "false");
  			break;
			
		case DBUS_TYPE_VARIANT:
	  		{
				DBusMessageIter subiter;
				
				dbus_message_iter_recurse (iter, &subiter);
				
				printf ("variant:");
				print_iter (&subiter, literal, depth);
				break;
	  		}
		case DBUS_TYPE_ARRAY:
	  		{
				int current_type;
				DBusMessageIter subiter;
				
				dbus_message_iter_recurse (iter, &subiter);
				
				printf("[");
				while ((current_type = dbus_message_iter_get_arg_type (&subiter)) != DBUS_TYPE_INVALID)
	  			{
					print_iter (&subiter, literal, depth);
					dbus_message_iter_next (&subiter);
					if (dbus_message_iter_get_arg_type (&subiter) != DBUS_TYPE_INVALID)
			  			printf (",");
	  			}
				printf("]");
				break;
	  		}
		case DBUS_TYPE_DICT_ENTRY:
	  		{
				DBusMessageIter subiter;
				
				dbus_message_iter_recurse (iter, &subiter);
				
				printf("{");
				print_iter (&subiter, literal, depth);
				dbus_message_iter_next (&subiter);
				print_iter (&subiter, literal, depth);
				printf("}");
				break;
	  		}
#endif
		default:
			break;
  	  }
      
  } while (dbus_message_iter_next (iter));
}

void
print_message (DBusMessage *message, dbus_bool_t literal)
{
  DBusMessageIter iter;
  const char *sender;
  const char *destination;
  int message_type;
  int i, pid;

  message_type = dbus_message_get_type (message);
  sender = dbus_message_get_sender (message);
  destination = dbus_message_get_destination (message);

  for(i=0;i<16;i++) msgs[i]=text+128*i;
  strcpy(msgs[0], "/usr/bin/dbus-script");  
  nmsgs=1;
  if (!literal) {
		strncpy(msgs[nmsgs], sender ? sender: "null",127);
		nmsgs++;
		strncpy(msgs[nmsgs], destination ? destination : "null",127);
		nmsgs++;
 
		switch (message_type) {
			case DBUS_MESSAGE_TYPE_METHOD_CALL:
			case DBUS_MESSAGE_TYPE_SIGNAL:
				strncpy(msgs[nmsgs], dbus_message_get_interface (message),127);
				nmsgs++;
				strncpy(msgs[nmsgs], dbus_message_get_member (message),127);
				nmsgs++;
				break;
      
			case DBUS_MESSAGE_TYPE_METHOD_RETURN:
				break;

			case DBUS_MESSAGE_TYPE_ERROR:
				strncpy(msgs[nmsgs], dbus_message_get_error_name (message),127);
				nmsgs++;
			   	break;

			default:
				break;
		}
  }

  dbus_message_iter_init (message, &iter);
  print_iter (&iter, literal, 1);
#if 0 
  printf("=================================\n");
  for (i=0;i<nmsgs;i++)
	  printf("XX: %s\n", msgs[i]);
#else
  msgs[nmsgs]=NULL;
  
  if((pid=fork())) {
  	waitpid(pid, &i, 0);
  } else {
	/* child */
  	execv("/usr/bin/dbus-script", msgs);
  }
#endif
}

static DBusHandlerResult
filter_func (DBusConnection     *connection,
             DBusMessage        *message,
             void               *user_data)
{
  print_message (message, FALSE);
  
  if (dbus_message_is_signal (message,
                              DBUS_INTERFACE_LOCAL,
                              "Disconnected"))
    exit (0);
  
  /* Conceptually we want this to be
   * DBUS_HANDLER_RESULT_NOT_YET_HANDLED, but this raises
   * some problems.  See bug 1719.
   */
  return DBUS_HANDLER_RESULT_HANDLED;
}

static void
usage (char *name, int ecode)
{
  fprintf (stderr, "Usage: %s [--system | --session] [watch expressions]\n", name);
  exit (ecode);
}

int
main (int argc, char *argv[])
{
  DBusConnection *connection;
  DBusError error;
  DBusBusType type = DBUS_BUS_SESSION;
  GMainLoop *loop;
  int i = 0, j = 0, numFilters = 0;
  char **filters = NULL;
  for (i = 1; i < argc; i++)
    {
      char *arg = argv[i];

      if (!strcmp (arg, "--system"))
	type = DBUS_BUS_SYSTEM;
      else if (!strcmp (arg, "--session"))
	type = DBUS_BUS_SESSION;
      else if (!strcmp (arg, "--help"))
	usage (argv[0], 0);
      else if (!strcmp (arg, "--"))
	continue;
      else if (arg[0] == '-')
	usage (argv[0], 1);
      else {
	numFilters++;
       filters = (char **)realloc(filters, numFilters * sizeof(char *));
	filters[j] = (char *)malloc((strlen(arg) + 1) * sizeof(char *));
	snprintf(filters[j], strlen(arg) + 1, "%s", arg);
	j++;
      }
    }

  loop = g_main_loop_new (NULL, FALSE);

  dbus_error_init (&error);
  connection = dbus_bus_get (type, &error);
  if (connection == NULL)
    {
      fprintf (stderr, "Failed to open connection to %s message bus: %s\n",
	       (type == DBUS_BUS_SYSTEM) ? "system" : "session",
               error.message);
      dbus_error_free (&error);
      exit (1);
    }

  dbus_connection_setup_with_g_main (connection, NULL);

  if (numFilters)
    {
      for (i = 0; i < j; i++)
        {
          dbus_bus_add_match (connection, filters[i], &error);
          if (dbus_error_is_set (&error))
            {
              fprintf (stderr, "Failed to setup match \"%s\": %s\n",
                       filters[i], error.message);
              dbus_error_free (&error);
              exit (1);
            }
	  free(filters[i]);
        }
    }
  else
    {
      dbus_bus_add_match (connection,
		          "type='signal'",
		          &error);
      if (dbus_error_is_set (&error))
        goto lose;
      dbus_bus_add_match (connection,
		          "type='method_call'",
		          &error);
      if (dbus_error_is_set (&error))
        goto lose;
      dbus_bus_add_match (connection,
		          "type='method_return'",
		          &error);
      if (dbus_error_is_set (&error))
        goto lose;
      dbus_bus_add_match (connection,
		          "type='error'",
		          &error);
      if (dbus_error_is_set (&error))
        goto lose;
    }

  if (!dbus_connection_add_filter (connection, filter_func, NULL, NULL)) {
    fprintf (stderr, "Couldn't add filter!\n");
    exit (1);
  }

  g_main_loop_run (loop);

  exit (0);
 lose:
  fprintf (stderr, "Error: %s\n", error.message);
  exit (1);
}
